import yenv

from rest_framework.authentication import SessionAuthentication

from ok.core.permissions import is_yauser_authenticated


class OkAuthentication(SessionAuthentication):
    """
    DRF-аутентификация OK-а. Основана на сессионной аутентификации,
    отличается от неё тем, что при походах в OK через oauth или tvm,
    CSRF-защита отключается. В CSRF-защите нет необходимости, когда запрос
    отправляется не из браузера.

    Также, в отличие от стандартного поведения SessionAuthentication, проверяет CSRF,
    даже если пользователя нет в нашей БД, но он при этом авторизован (подробности в OK-163)
    """
    def authenticate(self, request):
        result = super().authenticate(request)
        if result is None and is_yauser_authenticated(request.yauser):
            self.enforce_csrf(request)
        return result

    def enforce_csrf(self, request):
        # Не проверяем CSRF в dev-окружениях
        if yenv.type == 'development':
            return

        mechanism = request.yauser.authenticated_by.mechanism_name

        # Не проверяем CSRF при авторизации по oauth и tvm
        if mechanism in {'oauth', 'tvm'}:
            return

        return super().enforce_csrf(request)
