from django.core.exceptions import PermissionDenied, ValidationError
from django.http import Http404
from rest_framework import exceptions, status
from rest_framework.response import Response
from rest_framework.views import set_rollback

from ok.utils.lock import CantTakeLockException


class EditException(Exception):
    pass


class ResponseError(Exception):
    """
    Ошибка, после которой нужно сразу вернуть ответ
    """
    def __init__(self, **kwargs):
        self.kwargs = kwargs

    @property
    def response(self):
        return Response(**self.kwargs)


class Http400(Exception):

    def __init__(self, code):
        self.code = code


class FlowDroppedHttp200(Exception):
    pass


class TableFlowCallFailed(Exception):

    def __init__(self, detail):
        self.detail = detail


class AlreadyExists(Exception):

    def __init__(self, message='already_exists', code='already_exists', params=None):
        self.message = message
        self.code = code
        self.params = params or {}


class SimpleValidationError(ValidationError):
    """
    ValidationError, у которой message == code
    """
    def __init__(self, code, params=None):
        super().__init__(code, code, params)


def exception_handler(exc, context):
    """
    За основу взят rest_framework.views.exception_handler

    Отличается тем, что мы расширяем формат ошибок:
    {
        "errors": {},  // формат sform
        "error": [
            {
                "code": "<tanker_code>",
                "message": "<человеко_читаемая_ошибка_для_нас>",
                "params": {}  // необязательный
            },
            ...
        ]
    }
    Если ошибка из sform, то придет errors.
    Если ошибка не из sform, то придет error.
    """
    handled = False
    headers = {}
    status_code = status.HTTP_400_BAD_REQUEST
    data = {}
    response = None

    if isinstance(exc, exceptions.APIException):
        headers = {}
        if getattr(exc, 'auth_header', None):
            headers['WWW-Authenticate'] = exc.auth_header
        if getattr(exc, 'wait', None):
            headers['Retry-After'] = '%d' % exc.wait

        if isinstance(exc.detail, (list, dict)):
            data = exc.detail
        else:
            data = {
                'error': [{
                    'code': exc.detail,
                }]
            }
        handled = True
        status_code = exc.status_code

    elif isinstance(exc, Http400):
        data = {
            'error': [{
                'code': exc.code,
            }]
        }
        handled = True
        status_code = status.HTTP_400_BAD_REQUEST

    elif isinstance(exc, PermissionDenied):
        data = {
            'error': [{
                'code': 'permission_denied',
            }]
        }
        handled = True
        status_code = status.HTTP_403_FORBIDDEN

    elif isinstance(exc, EditException):
        data = {
            'error': [{
                'code': 'edit error',
            }]
        }
        handled = True
        status_code = status.HTTP_400_BAD_REQUEST

    elif isinstance(exc, CantTakeLockException):
        data = {
            'error': [{
                'code': 'can\'t take lock',
            }]
        }
        handled = True
        status_code = status.HTTP_408_REQUEST_TIMEOUT

    elif isinstance(exc, Http404):
        data = {
            'error': [{
                'code': 'not_found',
            }]
        }
        handled = True
        status_code = status.HTTP_404_NOT_FOUND

    elif isinstance(exc, FlowDroppedHttp200):
        data = {
            'detail': {'error': 'Flow dropped'}
        }
        handled = True
        status_code = status.HTTP_200_OK
    elif isinstance(exc, AlreadyExists):
        data = {
            'error': [{
                'code': exc.code,
                'message': exc.message,
                'params': exc.params,
            }]
        }
        handled = True
        status_code = status.HTTP_400_BAD_REQUEST
    elif isinstance(exc, TableFlowCallFailed):
        data = {
            'detail': exc.detail
        }
        handled = True
        status_code = exc.detail['code']

        if status_code >= 500:
            status_code = status.HTTP_502_BAD_GATEWAY

    elif isinstance(exc, ResponseError):
        response = exc.response

    if handled:
        set_rollback()
        response = Response(data, status=status_code, headers=headers)

    return response
