import json
import sform

from django.core.exceptions import ValidationError, FieldDoesNotExist
from django.core.validators import validate_slug
from django.utils.translation import get_language

from ok.api.core.errors import SimpleValidationError
from ok.scenarios.models import Scenario
from ok.staff.models import Group
from ok.tracker.models import Queue


class JsonField(sform.Field):

    def clean(self, new_value, old_value, required, trim_on_empty, base_initial, base_data):
        if isinstance(new_value, (dict, list)):
            return new_value
        elif not new_value:
            if required:
                raise ValidationError(
                    'Required',
                    code='required',
                )
            return None
        elif isinstance(new_value, str):
            try:
                return json.loads(new_value)
            except json.JSONDecodeError:
                raise ValidationError(
                    'Value should be a serializable json string',
                    code='should_be_json',
                )


class SlugField(sform.CharField):

    type_name = 'char'

    def clean(self, new_value, *args, **kwargs):
        value = super().clean(new_value, *args, **kwargs)
        if value:
            value = value.lower()
            validate_slug(value)
        return value


class UniqueSlugField(SlugField):

    def __init__(self, model_class, model_field='slug', *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.model_class = model_class
        self.model_field = model_field

    def clean(self, new_value, *args, **kwargs):
        value = super().clean(new_value, *args, **kwargs)
        filter_params = {self.model_field: value}
        if self.model_class and self.model_class.objects.filter(**filter_params).exists():
            raise SimpleValidationError('already_exists')
        return value


class MultipleSuggestField(sform.MultipleSuggestField):

    type_name = 'multiplesuggest'

    def get_label_fields(self, label_fields=None):
        label_fields = label_fields or self.label_fields
        model = self.get_queryset().model
        default_lang = 'ru'
        lang = get_language() or default_lang

        fields = (
            label_fields
            if isinstance(label_fields, (list, tuple))
            else [label_fields]
        )

        label_fields = []
        for field in fields:
            field_name = f'{field}_{lang}'
            try:
                model._meta.get_field(field_name)
            except FieldDoesNotExist:
                label_fields.append(field)
            else:
                label_fields.append(field_name)
        return label_fields


class QueueChoiceField(sform.ModelChoiceField):

    type_name = 'choice'

    def __init__(self, **kwargs):
        kwargs.setdefault('to_field_name', 'name')
        kwargs.setdefault('label_extractor', lambda x: x.name)
        kwargs.setdefault('queryset', Queue.objects.all())
        super().__init__(**kwargs)

    def clean(self, new_value, *args, **kwargs):
        new_value = self.prepare_value(new_value)
        return super().clean(new_value, *args, **kwargs)

    def prepare_value(self, value):
        if isinstance(value, str):
            value = value.upper()
        return value


class QueueMultipleChoiceField(QueueChoiceField, sform.ModelMultipleChoiceField):

    type_name = 'multiplechoice'

    def prepare_value(self, value):
        if isinstance(value, (list, tuple)):
            value = [super(QueueMultipleChoiceField, self).prepare_value(v) for v in value]
        return value


class ScenarioChoiceField(sform.ModelChoiceField):

    type_name = 'choice'

    def __init__(self, **kwargs):
        kwargs.setdefault('to_field_name', 'slug')
        kwargs.setdefault('label_extractor', lambda x: x.name)
        kwargs.setdefault('queryset', Scenario.objects.all())
        super().__init__(**kwargs)


class ScenarioMultipleChoiceField(ScenarioChoiceField, sform.ModelMultipleChoiceField):

    type_name = 'multiplechoice'


class GroupMultipleSuggestField(MultipleSuggestField):

    def __init__(self, **kwargs):
        kwargs.setdefault('to_field_name', 'url')
        kwargs.setdefault('label_fields', {
            'caption': ['name'],
            'extra_fields': ['type', 'staff_id'],
        })
        kwargs.setdefault('queryset', Group.objects.all())
        super().__init__(**kwargs)
