import sform

from itertools import chain

from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError

from ok.tracker.queues import get_or_create_queue_by_object_id


User = get_user_model()

EMPTY_LABEL = '\u2014'
PG_MAX_INTEGER_VALUE = 2147483647
STAFF_USER_EXTERNAL = 'external'


class BaseForm(sform.SForm):
    """
    Базовая форма ОКа с некоторыми надстройками над SForm
    """
    # Словарь вида {field: state}, который переопределяет состояния полей
    FIELDS_STATE = {}

    def prepare_fields_state(self):
        fields_state = super().prepare_fields_state()
        fields_state |= self.FIELDS_STATE
        return fields_state


class InvalidLoginsError(ValidationError):

    def __init__(self, invalid_logins=None):
        key = 'invalid_logins'
        params = {key: list(invalid_logins or [])}
        super(InvalidLoginsError, self).__init__(key, key, params)


class UserPermissionsError(ValidationError):

    def __init__(self, external_users=None):
        key = 'external_users'
        params = {key: external_users}
        super(UserPermissionsError, self).__init__(key, key, params)


# TODO OK-1277: переделать это всё на нормальные поля формы
#  – теперь это сделать проще, потому что мы смотрим в БД, а не в staff-api
class UserValidationFormMixin(object):
    """
    Миксин для формы, который позволяет валидировать все поля,
    которые принимают пользователей, за 1 поход в staff-api.
    Нужно в подклассе в `USER_FIELDS` перечислить поля, которые ожидают
    пользователей (логины), и в методе `clean` вызвать `validate_user_fields`
    """
    USER_FIELDS = ()

    def _collect_logins(self):
        result = {}
        for field in self.USER_FIELDS:
            logins = self.cleaned_data.get(field)
            if logins:
                logins = [logins] if isinstance(logins, str) else logins
                result[field] = logins
        return result

    def _get_valid_users(self, logins):
        logins = [i.lower() for i in logins]
        return dict(
            User.objects
            .filter(username__in=logins)
            .values_list('username', 'affiliation')
        )

    def validate_user_fields(self, check_tracker=True):
        field_to_logins = self._collect_logins()
        raw_logins = set(chain.from_iterable(field_to_logins.values()))
        logins_to_affiliation = self._get_valid_users(raw_logins)
        valid_logins = list(logins_to_affiliation.keys()) + ['']

        if check_tracker:
            queue = get_or_create_queue_by_object_id(self.data.get('object_id'))
            allow_externals = queue.allow_externals if queue else True
        else:
            allow_externals = True

        errors: dict[tuple, list[ValidationError]] = {}
        for field, logins in field_to_logins.items():
            invalid_logins = set(logins) - set(valid_logins)
            if invalid_logins:
                errors[(field,)] = [InvalidLoginsError(invalid_logins)]
                continue

            if not allow_externals:
                external_users = [
                    login for login in logins
                    if login and logins_to_affiliation[login] == STAFF_USER_EXTERNAL
                ]
                if external_users:
                    errors.setdefault((field,), []).append(UserPermissionsError(external_users))

        if errors:
            raise ValidationError(errors)
