import sform

from django.conf import settings
from django.contrib.auth import get_user_model
from rest_framework import status
from rest_framework.generics import GenericAPIView
from rest_framework.response import Response

from ok.api.approvements.forms import ApprovementParentStageForm, JsonField
from ok.api.core.forms import BaseForm
from ok.api.core.views import LogContextMixin, BaseViewSet
from ok.approvements.controllers import enrich_stages_approvers
from ok.flows.executor import execute, execute_code
from ok.flows.models import Flow


class FlowTestView(LogContextMixin, GenericAPIView):

    model_class = Flow
    queryset = Flow.objects.all()

    def get(self, request, *args, **kwargs):
        flow = self.get_object()
        result = execute(flow, request.GET)
        return Response(result['data'])


class TestRequestForm(BaseForm):

    flow_code = sform.CharField(state=sform.REQUIRED)
    flow_context = JsonField(state=sform.REQUIRED)


class TestResultForm(BaseForm):

    stages = sform.GridField(
        field_instance=sform.FieldsetField(ApprovementParentStageForm),
    )
    traceback = sform.CharField()


class TestView(BaseViewSet):

    validator_class = TestRequestForm

    def post(self, request, *args, **kwargs):
        has_permission = (
            get_user_model().objects
            .filter(
                username=request.yauser.login,
                groups__name=settings.FLOW_USER_GROUP,
            )
        )
        if not has_permission:
            return Response({}, status=status.HTTP_403_FORBIDDEN)
        validator = self.get_validator_object(request.data)
        self.validate(validator)

        res = execute_code(
            code=validator.cleaned_data['flow_code'],
            context=validator.cleaned_data['flow_context']
        )
        res['stages'] = res['data'].pop('stages')
        if res.get('stages'):
            res['stages'] = enrich_stages_approvers(res['stages'], strict=False)

        if res.get('detail', {}).get('error') == 'Table flow call failed':
            code = res['detail']['code']
            if code >= 500:
                code = status.HTTP_502_BAD_GATEWAY

            return Response(res, status=code)
        return Response(res)
