import sform

from urllib.parse import urlparse, parse_qs

from django.utils.datastructures import MultiValueDict
from model_utils import Choices

from ok.api.approvements.forms import ApprovementCreateForm
from ok.api.core.errors import SimpleValidationError
from ok.api.core.fields import UniqueSlugField, GroupMultipleSuggestField
from ok.api.core.forms import BaseForm
from ok.api.core.validators import clean_staff_groups
from ok.api.core.views import unfold_query_params
from ok.api.scenarios.validators import validate_user_can_setup_queue
from ok.scenarios.choices import SCENARIO_STATUSES
from ok.scenarios.models import Scenario
from ok.tracker.choices import TRACKER_VARIABLES


class EditableFieldStateMixin:

    def get_field_state(self, name):
        if not self.base_initial.get('is_editable'):
            return sform.READONLY
        return super().get_field_state(name)


class ApprovementForScenarioForm(ApprovementCreateForm):

    FIELDS_STATE = {
        # Поля, которые в этой форме не нужны совсем
        'object_id': sform.READONLY,
        'scenario': sform.READONLY,
        'uid': sform.READONLY,
        'users': sform.READONLY,
        # Поля, которые пока что не нужны
        'author': sform.READONLY,
        'callback_url': sform.READONLY,
        'create_comment': sform.READONLY,
        'flow_context': sform.READONLY,
        'flow_name': sform.READONLY,
        'tvm_id': sform.READONLY,
    }


class ApprovementForScenarioUpdateForm(EditableFieldStateMixin, ApprovementForScenarioForm):
    pass


class ApprovementMacroCompatibilityForm(ApprovementForScenarioForm):

    _url_path = sform.ChoiceField(Choices('/tracker'))
    _embedded = sform.ChoiceField(Choices('1'))
    author = sform.ChoiceField(Choices(TRACKER_VARIABLES.current_user_login))
    object_id = sform.ChoiceField(Choices(TRACKER_VARIABLES.issue_key))
    uid = sform.ChoiceField(Choices(
        TRACKER_VARIABLES.current_datetime,
        TRACKER_VARIABLES.current_datetime_iso,
    ))
    scenario = sform.ChoiceField(Choices())

    # Note: в макросах может быть не только name и body, но и, например, issueUpdate.
    # Также у макроса может быть слишком длинное название, например.
    # Сейчас мы не готовы работать с такими макросами, но если поймём, что таких особо и нет,
    # можно будет убрать валидацию, а вместе с ней и лишние походы в Трекер из админки
    # https://a.yandex-team.ru/review/2554522/details#comment-3526109
    _macro_fields = sform.MultipleChoiceField(Choices('self', 'id', 'queue', 'name', 'body'))
    _macro_name = sform.CharField(max_length=255)

    # Автора отдельно валидировать не нужно, остаётся только users
    USER_FIELDS = (
        'users',
    )
    FIELDS_STATE = {
        '_url_path': sform.REQUIRED,
        '_macro_fields': sform.REQUIRED,
        '_macro_name': sform.REQUIRED,
        '_embedded': sform.REQUIRED,
        'author': sform.REQUIRED,
        'object_id': sform.REQUIRED,
        'uid': sform.REQUIRED,
        'scenario': sform.REQUIRED,

        'callback_url': sform.READONLY,
        'create_comment': sform.READONLY,
        'flow_context': sform.READONLY,
        'flow_name': sform.READONLY,
        'tvm_id': sform.READONLY,
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # Note: Чтобы наши собственные макросы воспринимались как совместимые, смотрим на scenario,
        # заданный в base_initial. Если scenario там задан, то scenario из макроса должен строго
        # с ним совпадать. Если не задан, значит мы совсем не знаем о таком макросе, и параметра
        # scenario в нём не должно быть в принципе
        if self.base_initial.get('scenario'):
            self.fields['scenario'].choices = Choices(self.base_initial['scenario'])
        else:
            self.fields.pop('scenario')

    @classmethod
    def init(cls, url, tracker_macro, *args, **kwargs):
        parsed = urlparse(url)
        data = MultiValueDict(parse_qs(parsed.query))
        data = unfold_query_params(data, ('users', 'groups'))
        data.setdefault('_url_path', parsed.path.rstrip('/'))
        data.setdefault('_macro_fields', [k for k, v in tracker_macro._value.items() if v])
        data.setdefault('_macro_name', tracker_macro.name)
        return cls(data, *args, **kwargs)

    def clean(self):
        # Проверяем, что все поля строго из списка допустимых
        readonly_fields = {k for k, v in self.FIELDS_STATE.items() if v == sform.READONLY}
        available_fields = set(self.fields) - readonly_fields
        unknown_fields = list(set(self.data) - available_fields)
        if unknown_fields:
            raise SimpleValidationError(
                code='unknown_fields',
                params=unknown_fields,
            )
        return super().clean()

    def clean_object_id(self, object_id):
        return object_id

    def validate_tracker_fields(self):
        pass


class ScenarioCreateValidateForm(BaseForm):

    slug = UniqueSlugField(model_class=Scenario, max_length=255)
    name = sform.CharField(max_length=255)
    responsible_groups = GroupMultipleSuggestField()
    tracker_queue = sform.CharField(max_length=15)
    approvement_data = sform.FieldsetField(ApprovementForScenarioForm)

    def clean_responsible_groups(self, groups):
        return clean_staff_groups(groups)

    def clean_tracker_queue(self, queue):
        if not queue:
            return queue
        queue = queue.strip().upper()
        validate_user_can_setup_queue(queue)
        return queue

    def clean_approvement_data(self, data):
        # После успешной валидации отдаём сырые данные, чтобы сохранить в БД как есть
        return self.data.get('approvement_data') or {}


class ScenarioCreateForm(ScenarioCreateValidateForm):

    FIELDS_STATE = {
        'slug': sform.REQUIRED,
        'name': sform.REQUIRED,
        'tracker_queue': sform.REQUIRED,
    }


class ScenarioUpdateForm(EditableFieldStateMixin, ScenarioCreateForm):

    approvement_data = sform.FieldsetField(ApprovementForScenarioUpdateForm)

    FIELDS_STATE = ScenarioCreateForm.FIELDS_STATE | {
        'slug': sform.READONLY,
        'tracker_queue': sform.READONLY,
    }


class ScenarioListFilterForm(BaseForm):

    statuses = sform.MultipleChoiceField(
        choices=SCENARIO_STATUSES,
        default=[SCENARIO_STATUSES.active],
        state=sform.REQUIRED,
    )
