import copy
import waffle

from django.utils.translation import gettext_lazy as _
from rest_framework import serializers

from ok.api.core.serializers import WorkflowActionsField
from ok.approvements.controllers import enrich_stages_approvers
from ok.scenarios.models import Scenario
from ok.scenarios.workflow import ScenarioWorkflow


class ScenarioSerializer(serializers.ModelSerializer):

    tracker_queue = serializers.ReadOnlyField(source='tracker_macro.tracker_queue.name')
    actions = WorkflowActionsField(workflow_class=ScenarioWorkflow)

    class Meta:
        model = Scenario
        fields = (
            'slug',
            'name',
            'status',
            'tracker_queue',
            'actions',
        )


class ScenarioUpdateFormSerializer(serializers.ModelSerializer):

    tracker_queue = serializers.ReadOnlyField(source='tracker_macro.tracker_queue.name')
    responsible_groups = serializers.SerializerMethodField()
    approvement_data = serializers.SerializerMethodField()

    def get_responsible_groups(self, obj):
        return [group.url for group in obj.responsible_groups.all()]

    def get_approvement_data(self, obj):
        data = copy.deepcopy(obj.approvement_data)
        if data.get('stages'):
            data['stages'] = enrich_stages_approvers(data['stages'], strict=False)
        return data

    class Meta:
        model = Scenario
        fields = (
            'slug',
            'name',
            'tracker_queue',
            'approvement_data',
            'responsible_groups',
        )


class ScenarioCloneFormSerializer(ScenarioUpdateFormSerializer):

    slug = serializers.SerializerMethodField()
    name = serializers.SerializerMethodField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._copy_number = None

    def copy_number(self, slug):
        if self._copy_number:
            return self._copy_number

        base_slug = f'{slug}-copy'
        slug_rgx = rf'^{base_slug}(-\d+)?$'
        copy_slugs = set(
            Scenario.objects
            .filter(slug__iregex=slug_rgx)
            .values_list('slug', flat=True)
        )

        copy_number = 1
        new_slug = base_slug
        while True:
            if new_slug not in copy_slugs:
                break
            copy_number += 1
            new_slug = f'{base_slug}-{copy_number}'
        self._copy_number = copy_number

        return self._copy_number

    def get_slug(self, obj):
        if not waffle.switch_is_active('enable_scenario_clone_slug_generation'):
            return ''
        slug = f'{obj.slug}-copy'
        copy_number = self.copy_number(obj.slug)
        if copy_number > 1:
            slug += f'-{copy_number}'
        return slug

    def get_name(self, obj):
        copy = _('copy')
        name = f'{obj.name} - {copy}'
        copy_number = self.copy_number(obj.slug)
        if copy_number > 1:
            name += f' {copy_number}'
        return name
