from django.db.models import Q
from rest_framework.response import Response

from ok.api.core.views import BaseViewSet, WorkflowView, unfold_query_params
from ok.api.scenarios import forms, serializers
from ok.scenarios.controllers import ScenarioController
from ok.scenarios.models import Scenario
from ok.scenarios.workflow import ScenarioWorkflow
from ok.staff.models import GroupMembership


_scenario_prefetch = (
    'responsible_groups__memberships',
    'tracker_macros__tracker_queue',
)


class ScenarioListCreateView(BaseViewSet):

    model_class = Scenario
    detail_serializer_class = serializers.ScenarioSerializer
    list_item_serializer_class = serializers.ScenarioSerializer
    only_validate = False

    def get_validator_class(self):
        if self.only_validate:
            return forms.ScenarioCreateValidateForm
        return forms.ScenarioCreateForm

    def get_queryset(self):
        return (
            super().get_queryset()
            .prefetch_related(*_scenario_prefetch)
        )

    def filter_queryset(self, queryset):
        params = unfold_query_params(
            query_params=self.request.query_params,
            list_fields=['statuses'],
        )
        filter_form = forms.ScenarioListFilterForm(data=params)
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        login = self.request.yauser.login
        groups = (
            GroupMembership.objects
            .filter(login=login)
            .values_list('group', flat=True)
        )
        return (
            queryset
            .filter(status__in=filter_params['statuses'])
            .filter(
                Q(author=login)
                | Q(responsible_groups__in=groups)
            )
            .order_by('-created')
            .distinct()
        )

    def validate_for_create(self, request, *args, **kwargs):
        self.only_validate = True
        validator = self.get_validator_object(request.data)
        self.validate(validator)
        return Response({})

    def perform_create(self, data):
        queue_name = data.pop('tracker_queue')
        scenario = ScenarioController.create(data, self.request.yauser.login)
        controller = ScenarioController(scenario)
        if queue_name:
            controller.create_macro(queue_name)
        return scenario


class ScenarioWorkflowView(WorkflowView):

    model_class = Scenario
    detail_serializer_class = serializers.ScenarioSerializer
    workflow_class = ScenarioWorkflow

    def get_queryset(self):
        return (
            super().get_queryset()
            .prefetch_related(*_scenario_prefetch)
        )


class ScenarioListFilterFormView(BaseViewSet):

    model_class = Scenario
    validator_class = forms.ScenarioListFilterForm

    def get_initial_data(self):
        return unfold_query_params(
            query_params=self.request.query_params,
            list_fields=['statuses'],
        )


class ScenarioDetailView(ScenarioWorkflowView):

    action_name = 'update'
    validator_class = forms.ScenarioUpdateForm
    form_serializer_class = serializers.ScenarioUpdateFormSerializer

    def get_validator_object(self, *args, **kwargs):
        base_initial = kwargs.setdefault('base_initial', {})
        base_initial['is_editable'] = self.workflow_action.is_available()
        return super().get_validator_object(*args, **kwargs)


class ScenarioCloneFormView(BaseViewSet):

    model_class = Scenario
    validator_class = forms.ScenarioCreateForm
    form_serializer_class = serializers.ScenarioCloneFormSerializer
