# coding: utf-8
from django.contrib import admin, messages
from django.contrib.contenttypes.admin import GenericTabularInline
from django.contrib.contenttypes.models import ContentType
from django.template.defaultfilters import truncatechars
from django.urls import reverse

from ok.approvements import models
from ok.approvements.choices import APPROVEMENT_RESOLUTIONS
from ok.core.admin import format_url


def get_admin_history_url(obj):
    url = reverse('admin:approvements_approvementhistory_changelist')
    content_type = ContentType.objects.get_for_model(obj)
    return '{}?content_type={}&object_id={}'.format(url, content_type.pk, obj.pk)


class ApprovementStageInlineAdmin(admin.TabularInline):

    model = models.ApprovementStage
    ordering = (
        'position',
    )
    raw_id_fields = (
        'parent',
        'approvement',
    )
    readonly_fields = (
        'approved_by',
        'approvement_source',
        'created',
        'modified',
        'history',
    )
    extra = 0

    def history(self, obj):
        return format_url(get_admin_history_url(obj), 'История')


class ApprovementGroupInlineAdmin(admin.TabularInline):

    model = models.ApprovementGroup
    raw_id_fields = (
        'group',
    )
    extra = 0


class ApprovementHistoryInlineAdmin(GenericTabularInline):

    model = models.ApprovementHistory
    fields = (
        'created',
        'event',
        'user',
        'status',
        'resolution',
    )
    readonly_fields = (
        'created',
    )
    extra = 0


@admin.register(models.Approvement)
class ApprovementAdmin(admin.ModelAdmin):

    search_fields = (
        '=id',
        '=uuid',
        'author',
        'text',
        'uid',
        'object_id',
    )

    list_display = (
        'id',
        'uuid',
        'author',
        'short_text',
        'status',
        'is_approved',
        'url',
        'direct_url',
        'callback_url',
    )

    list_filter = (
        'status',
    )

    readonly_fields = (
        'created',
        'modified',
    )

    raw_id_fields = (
        'scenario',
        'tracker_queue',
    )

    inlines = (
        ApprovementStageInlineAdmin,
        ApprovementGroupInlineAdmin,
        ApprovementHistoryInlineAdmin,
    )

    actions = (
        'close',
    )

    def short_text(self, obj):
        return truncatechars(obj.text, 50)

    def url(self, obj):
        return format_url(obj.url, obj.object_id)

    def callback_url(self, obj):
        return format_url(obj.callback_url, 'Callback url')

    def direct_url(self, obj):
        return format_url(obj.direct_url, 'Фронт')

    def is_approved(self, obj):
        if obj.resolution not in APPROVEMENT_RESOLUTIONS:
            return None
        return obj.resolution == APPROVEMENT_RESOLUTIONS.approved

    def has_superuser_permission(self, request):
        return request.user.is_superuser

    def _perform_workflow_action(self, request, queryset, action_name, **params):
        """
        Выполняет для переданных объектов заданное workflow-действие.
        Действие выполняется без проверки прав пользователя, но с проверкой статуса
        """
        from ok.approvements.workflow import ApprovementWorkflow

        user = request.user.username
        total_count = 0
        success_count = 0
        for instance in queryset:
            workflow = ApprovementWorkflow(instance, user)
            action = workflow.get_action(action_name)
            total_count += 1
            if action.is_status_correct():
                action.perform(**params)
                success_count += 1
        messages.info(request, f'Выполнено успешно: {success_count} / {total_count}')

    def close(self, request, queryset):
        self._perform_workflow_action(request, queryset, 'close')

    is_approved.boolean = True
    close.allowed_permissions = ['superuser']


@admin.register(models.ApprovementStage)
class ApprovementStageAdmin(admin.ModelAdmin):

    search_fields = (
        '=id',
        '=approvement__id',
        'approver',
    )

    raw_id_fields = (
        'approvement',
        'parent',
    )

    list_display = (
        'id',
        'approvement_id',
        'approver',
        'is_approved',
        'position',
    )

    readonly_fields = (
        'created',
        'modified',
    )

    inlines = (
        ApprovementStageInlineAdmin,
        ApprovementHistoryInlineAdmin,
    )


@admin.register(models.ApprovementHistory)
class ApprovementHistoryAdmin(admin.ModelAdmin):

    list_display = (
        'created',
        'event',
        'user',
        'content_type',
        'object_id',
        'status',
        'resolution',
    )

    readonly_fields = (
        'created',
        'modified',
        'content_object_url',
    )

    def content_object_url(self, obj):
        url = reverse(
            'admin:{}_{}_change'.format(obj.content_type.app_label, obj.content_type.model),
            args=(obj.object_id,),
        )
        return format_url(url, str(obj.content_object))
