from typing import List

from django.conf import settings
from django.db.models import Q, QuerySet
from django_yauth.user import YandexUser

from ok.approvements.choices import APPROVEMENT_ROLES
from ok.approvements.models import ApprovementStage, ApprovementGroup
from ok.utils.cache import memoize


def prefetch_stage_parents(depth=None):
    depth = depth or (settings.APPROVEMENT_STAGE_MAX_DEPTH - 1)
    assert depth > 0
    return '__'.join('parent' for _ in range(depth))


def filter_approvements_by_roles(queryset: QuerySet, user: YandexUser, roles: List[str]):
    # TODO OK-1015: полностью переделать на фильтр по ролям координатора
    if not roles:
        roles = [r for r, _ in APPROVEMENT_ROLES]
    q = Q()
    if APPROVEMENT_ROLES.author in roles:
        q |= Q(author=user.login)
    if APPROVEMENT_ROLES.responsible in roles:
        q |= Q(id__in=(
            ApprovementGroup.objects
            .filter(group__memberships__login=user.login)
            .values('approvement_id')
        ))
    return queryset.filter(q)


def get_suitable_approvements(queryset, sort, statuses=None, queues=None, scenarios=None, **kwargs):

    if statuses:
        queryset = queryset.filter(status__in=statuses)

    if queues:
        queryset = queryset.filter(tracker_queue__in=queues)

    if scenarios:
        queryset = queryset.filter(scenario__in=scenarios)

    queryset = queryset.order_by(sort)

    return queryset.distinct()


@memoize(5 * 60)
def get_user_approvements_counts(login):
    stages = (
        ApprovementStage.objects
        .filter(approver=login)
        .values('approvement_id')
        .order_by()
        .distinct()
    )
    return {
        'current_count': stages.current().count(),
    }
