import json

from collections import defaultdict
from datetime import timedelta

from constance import config
from django.contrib.contenttypes.models import ContentType
from django.core.management import BaseCommand
from django.db.models import Count, Subquery, OuterRef

from django.utils import timezone

from ok.approvements.models import Approvement, ApprovementHistory
from ok.utils.datetime import get_beginning_of_moscow_day


class Command(BaseCommand):

    help = 'Generate APPROVEMENT_AUTHORS_WHITELIST_BY_TRACKER_QUEUE config'

    def add_arguments(self, parser):
        parser.add_argument('--days-delta', action='store', default=90)
        parser.add_argument('--oks-num', action='store', default=3)
        parser.add_argument('--dry-run', action='store_true')
        parser.add_argument('--append', action='store_true')

    def handle(self, *args, **options):
        days_delta = int(options['days_delta'])
        oks_num = int(options['oks_num'])
        today = get_beginning_of_moscow_day(timezone.now())
        delta_days_ago = today - timedelta(days=days_delta)
        content_type = ContentType.objects.get_for_model(Approvement)

        # Фильтруем пары очередь в Трекере - логин инициатора запуска согласования,
        # где инициатор создал больше oks_num согласований за последние days_delta дней
        queue_initiator_pairs = (
            Approvement.objects
            .filter(created__gte=delta_days_ago)
            .annotate(initiator=Subquery(
                ApprovementHistory.objects
                .filter(object_id=OuterRef('id'), content_type=content_type)
                .values('user')[:1]
            ))
            .values('tracker_queue__name', 'initiator')
            .annotate(count=Count('id'))
            .filter(tracker_queue__name__isnull=False, count__gt=oks_num)
            .values_list('tracker_queue__name', 'initiator')
        )
        if options['append']:
            whitelist_config = json.loads(config.APPROVEMENT_AUTHORS_WHITELIST_BY_TRACKER_QUEUE)
            logins_by_queue = defaultdict(list, whitelist_config)
        else:
            logins_by_queue = defaultdict(list)
        for queue, initiator in queue_initiator_pairs:
            if initiator not in logins_by_queue[queue]:
                logins_by_queue[queue].append(initiator)

        str_logins_by_queue = json.dumps(logins_by_queue, indent=4)
        if not options['dry_run']:
            setattr(config, 'APPROVEMENT_AUTHORS_WHITELIST_BY_TRACKER_QUEUE', str_logins_by_queue)
        else:
            print(str_logins_by_queue)

        print('APPROVEMENT_AUTHORS_WHITELIST_BY_TRACKER_QUEUE value updated')
