from django.db.models import QuerySet, Manager

from ok.approvements.choices import (
    APPROVEMENT_STAGE_STATUSES,
    APPROVEMENT_STAGE_ACTIVE_STATUSES,
)


class Django31OrderingMixin:
    """
    В django 3 Meta.ordering в моделях не будет использоваться,
    поэтому пока делаем это самостоятельно.
    Но правильнее оставить сортировку в коде явно только там, где она нужна
    """
    def get_queryset(self):
        qs = super().get_queryset()
        ordering = self.model._meta.ordering
        if ordering:
            return qs.order_by(*ordering)


class ApprovementStageQuerySet(QuerySet):

    def pending(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.pending)

    def current(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.current)

    def suspended(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.suspended)

    def approved(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.approved)

    def rejected(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.rejected)

    def cancelled(self):
        return self.filter(status=APPROVEMENT_STAGE_STATUSES.cancelled)

    def active(self):
        return self.filter(status__in=APPROVEMENT_STAGE_ACTIVE_STATUSES._db_values)

    def root(self):
        return self.filter(parent__isnull=True)

    def leaves(self):
        """
        Листья – стадии, у которых есть согласующие
        """
        return self.exclude(approver='')


ApprovementStageManagerBase = Manager.from_queryset(ApprovementStageQuerySet)


class ApprovementStageManager(Django31OrderingMixin, ApprovementStageManagerBase):
    pass


class ApprovementHistoryManager(Django31OrderingMixin, Manager):
    pass
