import json

from django.template.loader import render_to_string
from django.utils.decorators import decorator_from_middleware
from django.utils.deprecation import MiddlewareMixin

from ok.tracker.tasks import create_comment_task


class DebugOkByTrackerIssueMiddleware(MiddlewareMixin):
    """
    Middleware добавляет возможность отправки ошибок ручки создания согласования в коммент в тикет
    """
    def process_request(self, request):
        request._body = request.body

    @staticmethod
    def _bytes_to_pretty_json(b, encoding='utf-8'):
        string = b.decode(encoding)
        data = json.loads(string)
        return json.dumps(data, ensure_ascii=False, indent=4)

    def process_response(self, request, response):
        issue_key = request.headers.get('x-ok-tracker-debug')
        if not issue_key or response.status_code < 400 or not hasattr(request, '_body'):
            return response

        try:
            request_content = self._bytes_to_pretty_json(request._body)
        except json.JSONDecodeError:
            request_content = request._body.decode('utf-8')

        context = {
            'request_content': request_content,
            'response_content': self._bytes_to_pretty_json(response.content),
        }
        text = render_to_string('tracker/debug-comment.wiki', context)
        create_comment_task.delay(issue_key, text=text)

        return response


debug_view_by_tracker_issue = decorator_from_middleware(DebugOkByTrackerIssueMiddleware)
