import uuid

from typing import List

from django.conf import settings
from django.contrib.contenttypes.fields import GenericForeignKey, GenericRelation
from django.contrib.contenttypes.models import ContentType
from django.contrib.postgres.fields import ArrayField, JSONField
from django.db import models
from dirtyfields import DirtyFieldsMixin
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.utils.functional import cached_property
from model_utils.models import TimeStampedModel

from ok.approvements import choices
from ok.approvements.managers import (
    ApprovementStageManager,
    ApprovementStageQuerySet,
    ApprovementHistoryManager,
)
from ok.utils.context import request_context


def _uuid_hex():
    return uuid.uuid4().hex


class Approvement(DirtyFieldsMixin, TimeStampedModel):

    uuid = models.UUIDField(
        default=uuid.uuid4,
        unique=True,
        editable=False,
    )
    author = models.CharField(max_length=255)
    scenario = models.ForeignKey(
        to='scenarios.Scenario',
        on_delete=models.PROTECT,
        related_name='approvements',
        null=True,
        blank=True,
    )

    # FIXME: OK-949
    #  Пока поле используется как основное,
    #  но для некоторых задач информация стала дублироваться
    #  в отдельную модель ApprovementGroup.
    #  Со временем нужно полностью перейти на неё
    groups = ArrayField(
        models.CharField(max_length=255),
        null=True,  # Обратная совместимость на время выкатки
        blank=True,
        default=list,
        help_text='Ответственные за согласование. Список group.url со Стаффа',
    )

    text = models.TextField()
    status = models.CharField(
        max_length=32,
        choices=choices.APPROVEMENT_STATUSES,
        default=choices.APPROVEMENT_STATUSES.in_progress,
    )
    resolution = models.CharField(
        max_length=32,
        blank=True,
        choices=choices.APPROVEMENT_RESOLUTIONS,
    )
    is_reject_allowed = models.NullBooleanField(
        default=True,
        help_text='Разрешается ставить "Не ОК"',
    )

    is_auto_approving = models.NullBooleanField(
        default=False,
        help_text=(
            'Если согласование происходит в трекере,'
            'то стадии где необходимо согласование автора тикета окаются автоматически'
        )
    )

    uid = models.CharField(
        max_length=32,
        help_text=(
            'Идентификатор согласования, псевдо-уникальный в рамках объекта согласования. '
            'Например, таким образом мы различаем два согласования в рамках одного тикета. '
            'По факту, это чаще всего захэшированные дата и время'
        ),
        default=_uuid_hex,
    )
    type = models.CharField(
        max_length=32,
        choices=choices.APPROVEMENT_TYPES,
        # для обратной совместимости по-умолчанию создаём трекерные согласования
        default=choices.APPROVEMENT_TYPES.tracker,
        null=True,
    )
    # Идентификатор объекта согласования (для трекера -- ключ тикета)
    object_id = models.CharField(max_length=32)

    # Признак того, что процесс согласования идёт параллельно для всех согласующих.
    # Это означает, что при запуске, приостановке и отмене согласования,
    # отбивки шлём не последовательно, а сразу всем.
    is_parallel = models.BooleanField(default=False)
    history = GenericRelation('approvements.ApprovementHistory')

    # Поля специфичные только для согласований запущенных через Трекер
    tracker_comment_id = models.CharField(
        max_length=32,
        blank=True,
        null=True,
    )
    tracker_comment_short_id = models.IntegerField(blank=True, null=True)
    tracker_queue = models.ForeignKey(
        'tracker.Queue',
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    callback_url = models.CharField(
        max_length=1024,
        blank=True,
        null=True,
    )

    tvm_id = models.IntegerField(
        null=True,
        blank=True,
    )

    disapproval_reasons = ArrayField(
        models.CharField(max_length=255),
        null=True,
        blank=True,
        default=list,
    )

    chosen_disapproval_reason = models.CharField(
        max_length=255,
        null=True,
        blank=True,
        default=None,
    )

    info_mails_to = ArrayField(
        models.EmailField(),
        null=True,
        blank=True,
        default=list,
    )

    @property
    def url(self):
        if self.is_tracker_approvement:
            url_hash = f'#{self.tracker_comment_id}' if self.tracker_comment_id else ''
            url = f'{settings.TRACKER_URL}{self.object_id}{url_hash}'
        else:
            url = self.direct_url
        return url

    @property
    def direct_url(self):
        """
        Прямая ссылка на согласование
        """
        return f'{settings.OK_URL}approvements/{self.uuid}'

    @property
    def wf_iframe(self):
        return (
            '{{iframe src="%s?_embedded=1" '
            'frameborder=0 width=100%% height=400px}}'
            % self.direct_url
        )

    def __str__(self):
        return f'Approvement ({self.id}): {self.author}, {self.object_id}, {self.status}'

    @property
    def next_stages(self) -> ApprovementStageQuerySet:
        statuses = (
            choices.APPROVEMENT_STAGE_STATUSES.pending,
            choices.APPROVEMENT_STAGE_STATUSES.rejected,
            choices.APPROVEMENT_STAGE_STATUSES.suspended,
        )
        is_deputy_q = models.Q(
            parent__is_with_deputies=True,
            position__gt=models.F('parent__position') + models.F('parent__need_approvals'),
        )
        # Получаем все стадии в определённых статусах, кроме тех, которые считаются запасными
        stages = (
            self.stages
            .filter(status__in=statuses)
            .exclude(is_deputy_q)
        )

        if self.is_parallel:
            return stages
        first_stage = stages.first()
        if not first_stage:
            return self.stages.none()
        q = models.Q(id=first_stage.id)
        if first_stage.is_parent:
            q |= models.Q(parent=first_stage)
        return stages.filter(q)

    @cached_property
    def is_complex_approvement(self):
        """
        Является ли согласование сложным, т.е. со вложенными стадиями
        """
        return self.stages.filter(parent__isnull=False).exists()

    @property
    def is_tracker_approvement(self):
        return self.type == choices.APPROVEMENT_TYPES.tracker

    class Meta:
        unique_together = ('uid', 'object_id')


class ApprovementStage(DirtyFieldsMixin, TimeStampedModel):

    objects = ApprovementStageManager()

    approvement = models.ForeignKey(
        to=Approvement,
        on_delete=models.CASCADE,
        related_name='stages',
    )
    parent = models.ForeignKey(
        to='self',
        on_delete=models.CASCADE,
        related_name='stages',
        blank=True,
        null=True,
    )
    approver = models.CharField(
        max_length=255,
        blank=True,
        help_text='Кто должен поставить "ок"',
    )
    approved_by = models.CharField(
        max_length=255,
        blank=True,
        help_text='Кто поставил "ок" на самом деле',
    )
    approvement_source = models.CharField(
        max_length=32,
        choices=choices.APPROVEMENT_STAGE_APPROVEMENT_SOURCES,
        blank=True,
        help_text='Как именно получен "ок" (из api/по комментарию)',
    )
    is_approved = models.NullBooleanField()
    status = models.CharField(
        max_length=32,
        choices=choices.APPROVEMENT_STAGE_STATUSES,
        default=choices.APPROVEMENT_STAGE_STATUSES.pending,
        null=True,
    )
    position = models.IntegerField()

    need_all = models.NullBooleanField(
        default=None,
        help_text='Нужен ОК от всех участников стадии (работает для родительских стадий)',
    )

    need_approvals = models.IntegerField(
        default=None,
        help_text='Необходимое кол-во ОК, что бы окнулась группа',
        null=True,
    )

    is_with_deputies = models.BooleanField(
        default=None,
        help_text=(
            'Признак того, что в стадии указаны дополнительные участники, '
            'которые тоже могут согласовать стадию (работает для родительских стадий)'
        ),
        null=True,
    )

    history = GenericRelation('approvements.ApprovementHistory')

    class Meta:
        ordering = ['position']
        indexes = [
            models.Index(
                fields=['status', 'approver'],
                name='stage_status_approver_idx',
            ),
        ]

    @property
    def is_parent(self):
        """
        Является ли стадия родительской. Проверка на деле фиктивная, в идеале стоит проверить,
        есть ли у стадии дочерние, но такая проверка сильно дороже, при этом отстутсвие approver
        сейчас однозначно говорит о том, что есть дети.
        """
        return self.approver == ''

    @property
    def is_leaf(self):
        return not self.is_parent

    @property
    def is_deputy(self):
        return bool(
            self.parent
            and self.parent.is_with_deputies
            and self.position > (self.parent.position + self.parent.need_approvals)
        )

    def get_need_all(self):
        return len(self.stages.all()) == self.need_approvals

    @property
    def is_active(self):
        return self.status in choices.APPROVEMENT_STAGE_ACTIVE_STATUSES

    @property
    def is_finished(self):
        return self.status in choices.APPROVEMENT_STAGE_FINISHED_STATUS_SET


class ApprovementGroup(models.Model):

    approvement = models.ForeignKey(
        to=Approvement,
        on_delete=models.CASCADE,
        related_name='approvement_groups',
    )
    group = models.ForeignKey(
        to='staff.Group',
        on_delete=models.PROTECT,
        related_name='approvement_groups',
    )

    def __str__(self):
        return f'Approvement ({self.approvement_id}), Group ({self.group_id})'

    class Meta:
        unique_together = ('approvement', 'group')


class ApprovementHistory(TimeStampedModel):
    """
    История изменений в approvement и его стадиях
    """
    objects = ApprovementHistoryManager()

    event = models.CharField(choices=choices.APPROVEMENT_HISTORY_EVENTS, max_length=32)
    user = models.CharField(max_length=255, blank=True, null=True)

    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_id = models.PositiveIntegerField()
    content_object = GenericForeignKey()

    status = models.CharField(max_length=32, blank=True)
    resolution = models.CharField(max_length=32, blank=True)

    context = JSONField(blank=True, default=dict)

    @cached_property
    def type(self):
        return self.content_type.name.replace(' ', '_').lower()

    @cached_property
    def approvement(self):
        if self.type == 'approvement':
            return self.content_object
        elif self.type == 'approvement_stage':
            return self.content_object.approvement

    @cached_property
    def stage(self):
        if self.type == 'approvement_stage':
            return self.content_object

    class Meta:
        ordering = ['created']
        indexes = [
            models.Index(fields=['content_type', 'object_id', 'event']),
        ]

    def __str__(self):
        return f'[{self.event}] {self.content_type_id} {self.object_id}'


def create_history_entry(obj, event, user, **kwargs):
    create_history_entries([obj], event, user, **kwargs)


def create_history_entries(objects, event, user, **kwargs):
    events = []
    for obj in objects:
        events.append(ApprovementHistory(
            event=event,
            user=user,
            content_object=obj,
            status=obj.status,
            resolution=getattr(obj, 'resolution', ''),
            **kwargs
        ))
    ApprovementHistory.objects.bulk_create(events)


@receiver(post_save, sender=Approvement, dispatch_uid='save_approvement_history')
def save_approvement_history(sender, instance, created, **kwargs):
    context = {
        'referer': request_context.referer,
        'ok_session_id': request_context.ok_session_id,
        'flow_name': request_context.flow_name,
    }
    if created:
        # Первое событие записываем в историю при создании
        create_history_entry(
            obj=instance,
            event=choices.APPROVEMENT_HISTORY_EVENTS.status_changed,
            user=request_context.user,
            created=instance.created,
            modified=instance.created,
            context=context,
        )
    elif {'status', 'resolution'} & set(instance.get_dirty_fields()):
        create_history_entry(
            obj=instance,
            event=choices.APPROVEMENT_HISTORY_EVENTS.status_changed,
            user=request_context.user,
            context=context,
        )


@receiver(post_save, sender=ApprovementStage, dispatch_uid='save_approvement_stage_history')
def save_approvement_stage_history(sender, instance, **kwargs):
    if {'is_approved', 'status'} & set(instance.get_dirty_fields()):
        create_history_entry(
            obj=instance,
            event=choices.APPROVEMENT_HISTORY_EVENTS.status_changed,
            user=request_context.user,
        )
