import logging
import re

from urllib.parse import urlparse, parse_qs

from django.conf import settings
from django.utils.functional import cached_property
from rest_framework import serializers
from startrek_client.exceptions import StartrekError

from ok.approvements.choices import (
    APPROVEMENT_STATUSES,
    APPROVEMENT_STATUS_TRACKER_MAPPING,
    APPROVEMENT_RESOLUTION_TRACKER_MAPPING,
)
from ok.tracker.base import client
from ok.tracker.helpers import fetch_ok_iframe_tracker_url
from ok.utils.strings import str_to_md5


logger = logging.getLogger(__name__)


class IssueFieldEnum:

    ACCESS = 'access'
    APPROVEMENT_ID = 'approvementId'
    APPROVEMENT_STATUS = 'approvementStatus'
    APPROVERS = 'approvers'
    CURRENT_APPROVERS = 'currentApprovers'


class IssueApprovementSerializer(serializers.Serializer):
    """
    Сериализация согласования для отправки в поля Трекера
    """
    access = serializers.SerializerMethodField()
    approvementId = serializers.CharField(source='uuid', read_only=True)
    approvementStatus = serializers.SerializerMethodField()
    approvers = serializers.SerializerMethodField()
    currentApprovers = serializers.SerializerMethodField()

    def __init__(self, instance, **kwargs):
        fields = set(kwargs.pop('fields', []))
        super().__init__(instance, **kwargs)
        for field in list(self.fields):
            if field not in fields:
                self.fields.pop(field, None)

    @cached_property
    def _approvers(self):
        return list(
            self.instance.stages
            .leaves()
            .values_list('approver', flat=True)
        )

    def get_access(self, obj):
        return {
            'add': self._approvers,
        }

    def get_approvementStatus(self, obj):
        # Для завершённых согласований в Трекере есть 2 статуса,
        # отличимые по нашим резолюциям
        if obj.status == APPROVEMENT_STATUSES.closed:
            return APPROVEMENT_RESOLUTION_TRACKER_MAPPING[obj.resolution]
        else:
            return APPROVEMENT_STATUS_TRACKER_MAPPING[obj.status]

    def get_approvers(self, obj):
        return self._approvers

    def get_currentApprovers(self, obj):
        return [s.approver for s in obj.stages.current().leaves()]


def _fetch_approvement_uid_from_url(url):
    """
    Извлекает uid из нашего iframe-url'а
    """
    parsed = urlparse(url)
    query_params = parse_qs(parsed.query)
    return query_params['uid'][0] if query_params.get('uid') else None


def _fetch_approvement_uuid(text):
    """
    Извлекает из коммента uuid согласования.
    Если извлечь не удалось, отдает None.
    """
    # uuid – 1234abcd-1234-1234-1234-12345678abcd
    uuid_rgx = r'[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}'
    rgx = re.compile(
        rf'{{iframe src="{settings.OK_URL}approvements/(?P<uuid>{uuid_rgx}).*?".*?}}',
        flags=re.UNICODE,
    )
    match = rgx.search(text)
    return match.groupdict().get('uuid') if match else None


def _fetch_comments_from_issue(approvement):
    """
    Генерит комменты, в которых есть согласование approvement
    """
    issue = client.issues[approvement.object_id]
    for comment in issue.comments.get_all():
        url = fetch_ok_iframe_tracker_url(comment.text)
        if url:
            uid = _fetch_approvement_uid_from_url(url)
            if uid and str_to_md5(uid) == approvement.uid:
                yield comment
            continue

        uuid = _fetch_approvement_uuid(comment.text)
        if uuid == str(approvement.uuid):
            yield comment


def set_approvement_tracker_comment_id(approvement):
    """
    Устанавливает согласованию ID коммента в Трекере
    """
    comments = list(_fetch_comments_from_issue(approvement))
    if not comments:
        logger.warning('Tracker comment for approvement %s was not found', approvement.id)
        return
    elif len(comments) > 1:
        logger.warning('Found multiple Tracker comments for approvement %s', approvement.id)

    approvement.tracker_comment_short_id = comments[0].id
    approvement.tracker_comment_id = comments[0].longId
    approvement.save(update_fields=['tracker_comment_short_id', 'tracker_comment_id', 'modified'])


def collect_issues_for_approvements(approvements):
    """
    Собирает данные по тикетам, в которых проходят согласования `approvements`.
    Для каждого согласования в атрибут `issue` записывается полученнный тикет.
    На данный момент у тикетов есть только поля по умолчанию (id, key и т.д.)
    и одно явно запрошенное поле `summary` (название тикета).

    Если тикеты получить не удалось, в атрибут `issue` запишется None.
    """
    fields = ['summary']
    keys = [a.object_id for a in approvements]
    try:
        issues = client.issues.find(keys=keys, fields=fields)
    except StartrekError as exc:
        logger.warning('Could not get issues %s: %s', keys, exc)
        issues = []

    issues_map = {i.key: i for i in issues}

    for approvement in approvements:
        approvement.issue = issues_map.get(approvement.object_id)
