from collections import defaultdict

from django.conf import settings
from django.db.models import prefetch_related_objects
from django.utils.functional import cached_property

from ok.approvements import models
from ok.approvements.yt.serializers import ApprovementHistorySerializer
from ok.utils.yt import YTTable


class YTApprovementHistoryTable(YTTable):

    serializer = ApprovementHistorySerializer
    schema = serializer.Meta.yt_schema
    path = settings.YT_APPROVEMENT_HISTORY_TABLE_PATH

    @cached_property
    def _serializer_context(self):
        return {}

    def _prefetch_related(self, queryset):
        history_entries = list(
            queryset
            .select_related('content_type')
            .prefetch_related('content_object')
        )

        # Префетчим approvement для всех стадий
        stages = [i.content_object for i in history_entries if i.type == 'approvement_stage']
        prefetch_related_objects(stages, 'approvement')

        # Достаем отдельно всех согласующих для всех согласований
        approvement_ids = {i.object_id for i in history_entries if i.type == 'approvement'}
        approvement_ids |= {i.approvement_id for i in stages}
        approvers = (
            models.ApprovementStage.objects
            .filter(approvement__in=approvement_ids)
            .values_list('approvement_id', 'approver')
        )
        approvers_by_approvement = defaultdict(list)
        for approvement_id, approver in approvers:
            approvers_by_approvement[approvement_id].append(approver)
        self._serializer_context['approvers_by_approvement'] = approvers_by_approvement

        return history_entries

    def write(self, queryset):
        assert queryset.model is models.ApprovementHistory
        rows = self._prefetch_related(queryset)
        super().write(rows)

    def serialize(self, row):
        return self.serializer(row, context=self._serializer_context).data
