import csv
import json

from django.contrib import admin
from django.http import HttpResponse
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from pygments import highlight
from pygments.formatters.html import HtmlFormatter
from pygments.lexers.data import JsonLexer


def fieldset(name=None, fields=None, collapse=False):
    return (name, {
        'fields': fields or (),
        'classes': ('collapse',) if collapse else (),
    })


def format_url(url, name=None):
    name = name or url
    return format_html(f'<a target="_blank" href="{url}">{name}</a>')


def pretty_json(data):
    json_data = json.dumps(data, sort_keys=True, indent=2, ensure_ascii=False)
    formatter = HtmlFormatter(style='colorful')
    result = highlight(json_data, JsonLexer(), formatter)
    style = '<style>' + formatter.get_style_defs() + '</style><br>'
    return mark_safe(style + result)


def export_as_csv(modeladmin, request, queryset):
    meta = queryset.model._meta
    field_names = [field.name for field in meta.fields]

    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = f'attachment; filename={meta.app_label}_{meta.model_name}.csv'
    writer = csv.writer(response)

    writer.writerow(field_names)
    for row in queryset.values_list(*field_names):
        writer.writerow(row)

    return response


export_as_csv.short_description = 'Export as CSV'


admin.site.add_action(export_as_csv)
