from operator import attrgetter, itemgetter

from simple_history.exceptions import NotHistoricalModelError
from simple_history.utils import get_history_manager_for_model, bulk_create_with_history


def update_instance(instance, data):
    """
    Редактирует поля instance, перечисленные в data
    """
    # Не позволяем вызывать функцию для объектов, которых еще нет в базе (то есть нет id)
    assert instance.pk is not None

    update_fields = set()

    for attr_name, attr_value in data.items():
        if not hasattr(instance, attr_name):
            continue
        old_value = getattr(instance, attr_name)
        if old_value != attr_value:
            setattr(instance, attr_name, attr_value)
            update_fields.add(attr_name)

    # Если update_fields окажется пустым, то записи в базу не будет
    instance.save(update_fields=list(update_fields))
    return instance


def update_list_of_instances(model, queryset, data, delete_missing=True,
                             identifier=None, delete_func=None):
    """
    Редактирует список сущностей, накладывая изменения на queryset.
    Если в data есть сущность, соответствующая identifier,
    она будет отредактирована.
    Если такой нет, то она будет создана.
    Если сущность была в базе, но ее не оказалось в data, то смотрим на флаг delete_missing.
    Если он True, то удаляем эти сущности с помощью функции delete_func, иначе оставляем.
    """
    if identifier is None:
        identifier = ('id',)

    if delete_func is None:
        delete_func = lambda queryset: queryset.delete()

    get_key_by_attr = attrgetter(*identifier)
    get_key_by_item = itemgetter(*identifier)

    entities_map = {}  # маппинг идентификатор: инстанс для queryset
    missing_entities_ids = set()  # сущности из базы, которым не нашлось соответствия
    handled_keys = set()  # обработанные идентификаторы, чтобы апдейтить только первый
    for entity in queryset:
        entities_map[get_key_by_attr(entity)] = entity
        missing_entities_ids.add(entity.id)

    new_instances = []

    updated = 0
    for entity_data in data:
        try:
            key = get_key_by_item(entity_data)
        except KeyError:
            # Если не получилось достать идентификатор, то будем создавать сущность.
            # В случае с id это ок, а вот в случае сложных идентификаторов - вопрос.
            instance = None
        else:
            instance = entities_map.get(key)

        if instance is not None:
            if key not in handled_keys:
                updated += 1
                update_instance(instance, entity_data)
                missing_entities_ids.discard(instance.id)
                handled_keys.add(key)
        else:
            entity_data.pop('id', None)
            new_instances.append(model(**entity_data))

    if delete_missing and missing_entities_ids:
        deleted_instances = delete_func(queryset.filter(id__in=missing_entities_ids))
    else:
        deleted_instances = []

    try:
        get_history_manager_for_model(model)
    except NotHistoricalModelError:
        created_instances = model.objects.bulk_create(new_instances)
    else:
        created_instances = bulk_create_with_history(new_instances, model)

    return len(created_instances), updated, len(deleted_instances)
