import importlib
import logging
import time

from celery.app.task import Task
from django.conf import settings
from django.core.management.base import BaseCommand, CommandError

from ok.utils.lock import lock_manager


logger = logging.getLogger(__name__)


class Command(BaseCommand):

    help = 'Run celery task by name.'

    def add_arguments(self, parser):
        parser.add_argument('--task', required=True)
        parser.add_argument('-s', '--synchronous', action='store_true')
        parser.add_argument('--delay', default=10, type=int)
        parser.add_argument('--timeout', default=10, type=int)
        parser.add_argument('--block', action='store_true')

    def handle(self, *args, **options):
        task_name = options['task']
        module, task = task_name.rsplit('.', 1)
        module = importlib.import_module(module)
        task = getattr(module, task)

        if not isinstance(task, Task):
            raise CommandError('This is not a celery task.')

        self.run_task_with_lock(task, options)

    def run_task_with_lock(self, task, options):
        lock_name = '{env}.{task}'.format(
            env=settings.APP_ENV or 'UNKNOWN_ENVIRONMENT',
            task=options['task'],
        )

        logger.info('Acquiring lock `%s`', lock_name)
        with lock_manager.lock(lock_name, options['timeout'], options['block']) as acquired:
            if acquired:
                logger.info('Lock `%s` acquired', lock_name)
                self.run_task(task, options['synchronous'], options['delay'])
            else:
                logger.info('Lock `%s` NOT acquired', lock_name)

    def run_task(self, task, is_synchronous, delay):
        if is_synchronous:
            task()
            print('Task was executed synchronously')
        else:
            print('Putting task into a queue with delay...', end=' ')
            task.delay()
            # Чтобы lock не освобождался слишком быстро
            time.sleep(delay)
            print('OK')
