from typing import Union, AnyStr, Generator, Optional

from ok.flows.flow_functions import (
    extend_approvers,
    add_approver,
    add_approvers_group,
    remove_duplicates,
)
from ok.utils.cache import memoize
from ok.utils.staff import get_staff_repo


_group_repo = get_staff_repo('group')
_person_repo = get_staff_repo('person')


class _Stages(list):

    def append_persons(self, *args: Union['Person', str]):
        for person in args:
            add_approver(self, person)

    def append_group(self, persons: list[Union['Person', str]], need_all: bool, *,
                     is_with_deputies=False):
        add_approvers_group(self, persons, need_all, is_with_deputies=is_with_deputies)

    def extend_by_stages(self, stages: '_Stages', merge_to_right=False):
        extend_approvers(self, stages, merge_to_right)

    def remove_dupes(self, keep_firsts: bool):
        remove_duplicates(self, keep_firsts)

    def to_dict(self) -> dict:
        return {'stages': str(self)}


class Department:

    def __init__(self, department_url):
        self.url = department_url

    def __eq__(self, other) -> bool:
        return isinstance(other, Department) and self.url == other.url

    def __hash__(self) -> int:
        return hash(self.url)

    def __str__(self) -> str:
        return self.url

    def __repr__(self) -> str:
        return f'Department<{self.url}>'

    @memoize(seconds=120)
    def _get_data(self):
        api_resp = _group_repo.get_one({
            'department.url': self.url,
            '_fields': ','.join([
                'parent.url',
                'department.heads.role',
                'department.heads.person.login',
                'ancestors.department.heads.person.login',
                'ancestors.department.heads.role',
                'ancestors.department.url',
            ])
        })

        return api_resp

    @property
    def parent(self) -> 'Department' or None:
        parent_url = self._get_data().get('parent', {}).get('url')
        return Department(parent_url) if parent_url else None

    def get_ancestors(self, with_self=True) -> list['Department']:
        result = [self] if with_self else []
        for ancestor in reversed(self._get_data()['ancestors']):
            result.append(Department(ancestor['department']['url']))
        return result

    def _get_role_owners(self, role_name: str) -> Generator['Person', None, None]:
        for person_role in self._get_data()['department']['heads']:
            if person_role['role'] == role_name:
                yield Person(person_role['person']['login'])

    def _get_ancestors_role_owners(
            self, role_name: str, first: bool = False) -> Generator['Person', None, None]:
        first_found = False
        for department in reversed(self._get_data()['ancestors']):
            for head in department['department']['heads']:
                if head['role'] == role_name:
                    yield Person(head['person']['login'])
                    first_found = True
            if first and first_found:
                return

    @property
    def head(self) -> Optional['Person']:
        return next(self._get_role_owners('chief'), None)

    @property
    def hr_partners(self) -> list['Person']:
        hr_partners = list(self._get_role_owners('hr_partner'))
        if not hr_partners:
            hr_partners = list(self._get_ancestors_role_owners('hr_partner', True))
        return hr_partners

    @property
    def heads_chain(self) -> list['Person']:
        direct = self.head
        if direct is None:
            return []
        heads = [self.head]
        for head in self._get_ancestors_role_owners('chief'):
            heads.append(head)
        return heads


class Person:
    def __init__(self, login):
        self.login = login

    def __eq__(self, other) -> bool:
        return isinstance(other, Person) and self.login == other.login

    def __hash__(self) -> int:
        return hash(self.login)

    def __str__(self) -> str:
        return self.login

    def __repr__(self) -> str:
        return f'Person<{self.login}>'

    @memoize(seconds=120)
    def _get_data(self):
        api_resp = _person_repo.get_one({
            'login': self.login,
            '_fields': ','.join([
                'department_group.url',
                'hr_partners.login',
                'chief.login',
                'chiefs.login',
                'official.affiliation',
                'official.is_trainee',
            ])
        })

        return api_resp

    def _get_department_url(self) -> AnyStr:
        return self._get_data()['department_group']['url']

    @property
    def department(self) -> Department:
        return Department(self._get_department_url())

    @property
    def hr_partners(self) -> list['Person']:
        return [Person(it['login']) for it in self._get_data()['hr_partners']]

    @property
    def head(self) -> 'Person':
        login = self._get_data().get('chief', {}).get('login')
        return Person(login)

    @property
    def heads_chain(self) -> list['Person']:
        return [Person(chief['login']) for chief in self._get_data()['chiefs']]

    @property
    def department_chain(self) -> list[Department]:
        return self.department.get_ancestors(with_self=True)

    @property
    def is_external(self) -> bool:
        return self._get_data()['official']['affiliation'] == 'external'

    @property
    def is_intern(self) -> bool:
        return self._get_data()['official']['is_trainee']
