import re

from logging import getLogger

from django.contrib.auth import get_user_model
from django.utils.functional import cached_property
from django.utils.translation import gettext_lazy as _

from ok.approvements.tracker import collect_issues_for_approvements
from ok.notifications.base import BaseNotification, BaseNotificationSender
from ok.notifications.utils import get_message_id
from ok.utils import wiki
from ok.utils.attrs import get_attribute


User = get_user_model()

logger = getLogger(__name__)


def _get_approvement_message_id(approvement):
    return get_message_id('approvement', approvement.id, approvement.created)


class ApprovementBaseNotification(BaseNotification):

    subject_prefix = None

    @cached_property
    def default_language(self):
        cyrillic = re.compile(r'[а-яё]', re.IGNORECASE)
        return 'ru' if cyrillic.search(self.instance.text) else 'en'

    # FIXME: этот метод теперь ещё более не оптимален.
    #  Т.к. мы ходим в БД за пользователями при формировании контекста,
    #  с включением уведомлений для моб.приложения, мы теперь это делаем в 2 раза больше
    #  https://st.yandex-team.ru/OK-1264
    def _add_users_to_context(self, context):
        logins = (
            self.initiator,
            self.instance.author,
        )
        users_map = User.objects.in_bulk(logins, field_name='username')
        context['initiator'] = users_map.get(self.initiator)
        context['author'] = users_map.get(self.instance.author)

    def _get_root_stages(self):
        if hasattr(self.instance, 'root_stages'):
            return list(self.instance.root_stages)
        return list(self.instance.stages.root().prefetch_related('stages'))

    def get_context(self):
        collect_issues_for_approvements([self.instance])
        context = super().get_context()
        self._add_users_to_context(context)
        context['stages'] = list(self.instance.stages.all())
        context['root_stages'] = self._get_root_stages()
        context['formatted_text'] = wiki.format_markup_safe(
            text=self.instance.text,
            default=self.instance.text,
        )
        return context

    def get_subject(self):
        if self.subject:
            return str(self.subject)
        summary = get_attribute(self.instance, 'issue.summary', '')
        return f'[{self.subject_prefix}] ({self.instance.object_id}) {summary}'

    def get_url(self):
        return self.instance.url

    def get_thread_id(self):
        return _get_approvement_message_id(self.instance)


class ApprovementToResponsibleNotification(ApprovementBaseNotification):

    def get_receiver_logins(self) -> set[str]:
        # Note: В info_mails_to помимо рассылок могут оказаться почты других пользователей,
        # поэтому воспринимаем всё до `@` как логин пользователя,
        # чтобы получить для таких пользователей правильный язык
        # (см. BaseNotification.get_language_by_receiver_email)
        info_mails_to = self.instance.info_mails_to or []
        if info_mails_to:
            return {i.split('@')[0] for i in info_mails_to}
        return {self.instance.author}


class ApprovementExampleEmailNotification(ApprovementBaseNotification):
    """
    Просто пример нотификации для вёрстки писем
    """
    subject_prefix = _('EXAMPLE')
    template_name = 'email/approvements/example.html'

    def send(self, **kwargs):
        # Чтобы случайно не отправить однажды
        raise NotImplementedError


class ApprovementBulkNotificationBase(BaseNotification):

    def __init__(self, receiver, initiator=None, **kwargs):
        kwargs['receiver'] = receiver
        super().__init__(instance=None, initiator=initiator, **kwargs)

    def get_receiver_logins(self):
        return [self.kwargs['receiver']]


class ApprovementRequiredEmailNotification(ApprovementBaseNotification):

    subject_prefix = _('Approval action required')
    template_name = 'email/approvements/required.html'

    def get_receiver_logins(self):
        return {stage.approver for stage in self.kwargs['current_stages']}


class ApprovementRequiredAppNotification(ApprovementRequiredEmailNotification):

    transport = 'xiva'
    subject = ApprovementRequiredEmailNotification.subject_prefix
    template_name = 'mobile_app/approvements/required.txt'


class ApprovementRequiredNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementRequiredEmailNotification,
        ApprovementRequiredAppNotification,
    ]


class ApprovementApprovedByResponsibleEmailNotification(ApprovementBaseNotification):

    subject_prefix = _('Approved on your behalf')
    template_name = 'email/approvements/approved_by_responsible.html'

    def get_receiver_logins(self):
        return self.kwargs['receivers']


class ApprovementApprovedByResponsibleAppNotification(
    ApprovementApprovedByResponsibleEmailNotification,
):

    transport = 'xiva'
    subject = ApprovementApprovedByResponsibleEmailNotification.subject_prefix
    template_name = 'mobile_app/approvements/approved_by_responsible.txt'


class ApprovementApprovedByResponsibleNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementApprovedByResponsibleEmailNotification,
        ApprovementApprovedByResponsibleAppNotification,
    ]


class ApprovementFinishedEmailNotification(ApprovementToResponsibleNotification):

    subject_prefix = _('Approval process finished')
    template_name = 'email/approvements/finished.html'


class ApprovementFinishedAppNotification(ApprovementFinishedEmailNotification):

    transport = 'xiva'
    subject = ApprovementFinishedEmailNotification.subject_prefix
    template_name = 'mobile_app/approvements/finished.txt'


class ApprovementFinishedNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementFinishedEmailNotification,
        ApprovementFinishedAppNotification,
    ]


class ApprovementStoppedNotificationBase(ApprovementToResponsibleNotification):

    def get_receiver_logins(self) -> set[str]:
        logins = super().get_receiver_logins()
        logins |= {stage.approver for stage in self.kwargs['current_stages']}
        logins.discard(self.initiator)
        return logins


class ApprovementSuspendedEmailNotification(ApprovementStoppedNotificationBase):

    subject_prefix = _('Approval process suspended')
    template_name = 'email/approvements/suspended.html'


class ApprovementSuspendedAppNotification(ApprovementSuspendedEmailNotification):

    transport = 'xiva'
    subject = ApprovementSuspendedEmailNotification.subject_prefix
    template_name = 'mobile_app/approvements/suspended.txt'


class ApprovementSuspendedNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementSuspendedEmailNotification,
        ApprovementSuspendedAppNotification,
    ]


class ApprovementCancelledNotification(ApprovementStoppedNotificationBase):

    subject_prefix = _('Approval process cancelled')
    template_name = 'email/approvements/cancelled.html'


class ApprovementQuestionEmailNotification(ApprovementToResponsibleNotification):

    subject_prefix = _('Question about approval')
    template_name = 'email/approvements/question.html'


class ApprovementQuestionAppNotification(ApprovementQuestionEmailNotification):

    transport = 'xiva'
    subject = ApprovementQuestionEmailNotification.subject_prefix
    template_name = 'mobile_app/approvements/question.txt'


class ApprovementQuestionNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementQuestionEmailNotification,
        ApprovementQuestionAppNotification,
    ]


class ApprovementReminderEmailNotification(ApprovementBulkNotificationBase):

    subject = _('Approval processes await')
    template_name = 'email/approvements/reminder.html'

    def get_context(self):
        context = super().get_context()
        collect_issues_for_approvements(context['approvements'])
        return context


class ApprovementReminderAppNotification(ApprovementBulkNotificationBase):

    transport = 'xiva'
    subject = ApprovementReminderEmailNotification.subject
    template_name = 'mobile_app/approvements/reminder.txt'


class ApprovementReminderNotification(BaseNotificationSender):

    nested_notification_classes = [
        ApprovementReminderEmailNotification,
        ApprovementReminderAppNotification,
    ]


class ApprovementOverdueNotification(ApprovementBulkNotificationBase):

    subject = _('Approval awaits too long')
    template_name = 'email/approvements/overdue.html'

    def get_context(self):
        context = super().get_context()
        context['approvement_to_current_stages'] = dict(context['approvement_to_current_stages'])
        collect_issues_for_approvements(context['approvement_to_current_stages'])
        return context
