import waffle

from collections import defaultdict
from typing import Iterable

from django.conf import settings
from django.contrib.auth import get_user_model
from django.utils import translation
from django.utils.functional import cached_property

from ok.core.context import get_base_context
from ok.notifications import notify


User = get_user_model()


class BaseNotification(object):

    template_name = None
    default_language = 'en'
    transport = 'email'
    url = settings.OK_URL
    subject = None

    def __init__(self, instance, initiator=None, **kwargs):
        self.instance = instance
        self.initiator = initiator
        self.kwargs = kwargs

    def get_subject(self):
        return str(self.subject)

    def get_url(self):
        return self.url

    def get_receiver_logins(self) -> Iterable[str]:
        raise NotImplementedError

    @cached_property
    def receivers(self) -> Iterable[User]:
        return User.objects.filter(username__in=self.get_receiver_logins())

    def get_receiver_emails(self) -> set[str]:
        # Не берём почту из receivers, а формируем сами из логинов,
        # потому что у нас в качестве получателей могут быть рассылки,
        # для которых нет User (например, через info_mails_to)
        return {f'{login}@yandex-team.ru' for login in self.get_receiver_logins()}

    def get_thread_id(self):
        return None

    def get_template_name(self):
        return self.template_name

    def get_context(self):
        context = get_base_context()
        context['instance'] = self.instance
        context['initiator'] = self.initiator
        context.update(self.kwargs)
        return context

    def get_language_by_receiver_email(self) -> dict[str, str]:
        language_by_email = {r.email: r.language for r in self.receivers}
        return {
            email: language_by_email.get(email) or self.default_language
            for email in self.get_receiver_emails()
        }

    def send(self, **kwargs):
        if self.transport == 'email':
            self.send_email(**kwargs)
        elif self.transport == 'xiva':
            if waffle.switch_is_active('enable_mobile_app_notification'):
                self.send_mobile_app_notification(**kwargs)

    def send_email(self, **kwargs):
        context = self.get_context()
        template_name = self.get_template_name()

        kwargs.setdefault('initiator', self.initiator)
        kwargs.setdefault('message_id', None)
        kwargs.setdefault('in_reply_to', self.get_thread_id())

        if 'reply_to' not in kwargs and self.initiator:
            kwargs['reply_to'] = f'{self.initiator}@yandex-team.ru'

        for receiver, lang in self.get_language_by_receiver_email().items():
            with translation.override(lang):
                notify(
                    transport='email',
                    template_name=template_name,
                    context=context,
                    subject=self.get_subject(),
                    receiver=receiver,
                    **kwargs
                )

    def send_mobile_app_notification(self, **kwargs):
        context = self.get_context()
        template_name = self.get_template_name()

        kwargs.setdefault('event', self.__class__.__name__)

        uids_by_lang = defaultdict(list)
        for receiver in self.receivers:
            uids_by_lang[receiver.language].append(receiver.uid)

        data = dict(
            template_name=template_name,
            context=context,
            url=self.get_url(),
            **kwargs,
        )
        for lang, uids in uids_by_lang.items():
            with translation.override(lang):
                notify(transport='xiva', user_uids=uids, title=self.get_subject(), **data)

    def __call__(self, **kwargs):
        self.send(**kwargs)


class BaseNotificationSender:

    nested_notification_classes = []

    def __init__(self, *args, **kwargs):
        self.notification_instances = []
        for notification_class in self.nested_notification_classes:
            self.notification_instances.append(notification_class(*args, **kwargs))

    def send(self, **kwargs):
        for notification_instance in self.notification_instances:
            notification_instance.send(**kwargs)
