import logging
import smtplib
import yenv

from django.conf import settings
from django.core.mail import EmailMessage, get_connection
from django.core.mail.message import make_msgid

from ok.celery_app import app, get_retry_countdown
from ok.utils.xiva import push_notification_to_app


logger = logging.getLogger(__name__)


@app.task(max_retries=5)
def send_email(subject, body, to, **kwargs):
    subject = subject.replace('\r\n', ' ').replace('\n', ' ')
    original_to = list(to)

    headers = kwargs.setdefault('headers', {})
    from_email = kwargs.setdefault('from_email', settings.DEFAULT_FROM_EMAIL)
    message_id = kwargs.get('message_id')
    in_reply_to = kwargs.get('in_reply_to')
    reply_to = kwargs.get('reply_to')

    if yenv.type != 'production':
        to = [email for email in to if email in settings.DEBUG_EMAIL_WHITELIST]
        to.append(settings.DEBUG_EMAIL)
        body += '<br/>\nOriginal To: %s<br/>\nReal To: %s' % (original_to, to)

    # Сами генерируем Message-ID, чтобы в случае ретрая отправлялось то же самое письмо
    if 'Message-ID' not in headers:
        headers['Message-ID'] = message_id or make_msgid()

    if in_reply_to and 'In-Reply-To' not in headers:
        headers['In-Reply-To'] = in_reply_to

    if reply_to and 'Reply-To' not in headers:
        headers['Reply-To'] = reply_to

    connection = get_connection()
    try:
        connection.open()
    except Exception:
        logger.exception('Cannot reach EMAIL_BACKEND %s', settings.EMAIL_BACKEND)

    msg = EmailMessage(
        subject=subject,
        body=body,
        from_email=from_email,
        to=to,
        headers=headers,
        connection=connection,
    )
    msg.content_subtype = 'html'

    try:
        msg.send()
    except Exception as exc:
        if isinstance(exc, smtplib.SMTPServerDisconnected):
            logger.error('SMTP server disconnected, but email may be sent.')
            # Если словили эту ошибку, то ретраить будем максимум последний раз, чтобы не заспамить
            # https://st.yandex-team.ru/MAILADM-5207
            send_email.max_retries = 1

        logger.exception('Error during send email with Message-ID %s', headers['Message-ID'])
        send_email.retry(
            countdown=get_retry_countdown(send_email.request.retries),
            exc=exc,
            args=[subject, body, original_to],
            kwargs=kwargs,
        )
    else:
        logger.info('Sent email %(subject)s to %(to)s, Message-ID: %(message_id)s', {
            'subject': subject,
            'to': to,
            'message_id': headers['Message-ID'],
        })
    finally:
        connection.close()


@app.task
def push_notification_to_app_task(body, **kwargs):
    # таск не надо ретраить, иначе можем пользователям с несколькими подписками задублировать пуши
    kwargs['description'] = body
    push_notification_to_app(**kwargs)
