from django import http
from django.conf import settings
from django.core.mail import EmailMessage
from django.shortcuts import get_object_or_404
from django.utils.functional import cached_property
from django.views.generic.base import TemplateView
from premailer import Premailer

from ok.approvements.models import Approvement
from ok.notifications.approvements import (
    ApprovementQuestionEmailNotification,
    ApprovementReminderEmailNotification,
    ApprovementOverdueNotification,
)


def transform_callback(response):
    content = response.content.decode('utf-8')
    response.content = Premailer(content, remove_classes=True).transform()


class ApprovementEmailView(TemplateView):
    """
    Вьюха для вёрстки шаблонов писем
    для одного согласования
    """
    notification_class = None

    def get_extra_data(self):
        return {}

    @cached_property
    def notification(self):
        instance = get_object_or_404(Approvement, pk=self.kwargs['pk'])
        initiator = self.request.yauser.login
        return self.notification_class(instance, initiator, **self.get_extra_data())

    def get_template_names(self):
        return [self.notification.get_template_name()]

    def get_context_data(self, **kwargs):
        return dict(
            self.notification.get_context(),
            show_send_button=True,
        )

    def get(self, request, *args, **kwargs):
        response = super().get(request, *args, **kwargs)
        response.add_post_render_callback(transform_callback)
        return response

    def post(self, request, *args, **kwargs):
        response = self.get(request, *args, **kwargs)
        response.render()
        msg = EmailMessage(
            subject=self.notification.get_subject(),
            body=response.content.decode('utf-8'),
            from_email=settings.DEFAULT_FROM_EMAIL,
            to=[settings.DEBUG_EMAIL],
        )
        msg.content_subtype = 'html'
        msg.send()
        return http.HttpResponse('OK')


class ApprovementQuestionEmailView(ApprovementEmailView):

    notification_class = ApprovementQuestionEmailNotification

    def get_extra_data(self):
        return {
            'comment': 'Мне очень не нравится этот тикет',
        }


class ApprovementReminderEmailView(ApprovementEmailView):

    @cached_property
    def notification(self):
        approvements = Approvement.objects.order_by('-id')[:5]
        return ApprovementReminderEmailNotification(
            approvements=approvements,
            receiver=self.request.yauser.login,
        )


class ApprovementOverdueEmailView(ApprovementEmailView):

    @cached_property
    def notification(self):
        approvements = (
            Approvement.objects
            .prefetch_related('stages')
            .order_by('-id')[:5]
        )
        approvement_to_current_stages = {a: list(a.stages.all()) for a in approvements}
        return ApprovementOverdueNotification(
            approvement_to_current_stages=approvement_to_current_stages,
            receiver=self.request.yauser.login,
        )
