import base64
import json

from django.contrib import admin, messages
from django.forms import ModelForm, IntegerField, HiddenInput, ValidationError
from django.urls import reverse
from simple_history.admin import SimpleHistoryAdmin

from ok.api.scenarios.forms import (
    ApprovementMacroCompatibilityForm,
    ApprovementForScenarioForm,
)
from ok.core.admin import pretty_json, format_url
from ok.future.django.admin import EmptyFieldListFilter
from ok.scenarios import models
from ok.scenarios.controllers import ScenarioController
from ok.scenarios.tasks import sync_macros_task
from ok.tracker.helpers import fetch_ok_iframe_tracker_url


class CompatibilityError(ValidationError):

    def __init__(self):
        super().__init__('TrackerMacro is not compatible')


class ScenarioGroupInlineAdmin(admin.TabularInline):

    model = models.ScenarioResponsibleGroup
    raw_id_fields = (
        'group',
    )
    extra = 0


class ScenarioTrackerMacroInlineAdmin(admin.StackedInline):

    model = models.ScenarioTrackerMacro
    raw_id_fields = (
        'tracker_queue',
    )
    extra = 0


class ScenarioTrackerTriggerInlineAdmin(admin.StackedInline):

    model = models.ScenarioTrackerTrigger
    raw_id_fields = (
        'tracker_queue',
    )
    extra = 0


class ScenarioForm(ModelForm):

    macro_id = IntegerField(widget=HiddenInput(), required=False)

    def clean(self):
        cleaned_data = super().clean()
        macro_id = cleaned_data.get('macro_id')
        if macro_id:
            try:
                cleaned_data['macro'] = models.ScenarioTrackerMacro.objects.get(id=macro_id)
            except models.ScenarioTrackerMacro.DoesNotExist:
                raise ValidationError('TrackerMacro does not exist')

        macro = cleaned_data.get('macro')
        if not macro or self.instance.pk:
            return cleaned_data

        if macro.scenario_id:
            raise ValidationError('TrackerMacro has scenario already')

        if not macro.is_compatible:
            raise CompatibilityError()

        ok_tracker_url = fetch_ok_iframe_tracker_url(macro.body)
        if not ok_tracker_url:
            raise CompatibilityError()

        if not macro.remote_tracker_macro.macro:
            raise ValidationError(str(macro.remote_tracker_macro.exception))
        compatibility_form = ApprovementMacroCompatibilityForm.init(
            url=ok_tracker_url,
            tracker_macro=macro.remote_tracker_macro.macro,
        )
        if not compatibility_form.is_valid():
            raise CompatibilityError()

        return cleaned_data


@admin.register(models.Scenario)
class ScenarioAdmin(SimpleHistoryAdmin):

    form = ScenarioForm

    search_fields = (
        '=slug',
        'name',
        'author',
    )

    list_display = (
        'slug',
        'name',
        'author',
        'status',
    )

    list_filter = (
        'status',
    )

    inlines = (
        ScenarioTrackerMacroInlineAdmin,
        ScenarioTrackerTriggerInlineAdmin,
        ScenarioGroupInlineAdmin,
    )

    readonly_fields = (
        'pretty_approvement_data',
    )

    def pretty_approvement_data(self, obj):
        return pretty_json(obj.approvement_data)

    def get_changeform_initial_data(self, request):
        initial = super().get_changeform_initial_data(request)
        if 'approvement_data' in initial:
            data = initial['approvement_data']
            try:
                decoded_data = json.loads(base64.urlsafe_b64decode(data))
                initial['approvement_data'] = decoded_data
            except Exception:
                pass
        return initial

    def save_model(self, request, obj, form, change):
        super().save_model(request, obj, form, change)
        macro = form.cleaned_data.get('macro')
        if not macro or change:
            return
        macro.scenario = obj
        macro.save(update_fields=['scenario'])
        ctl = ScenarioController(obj)
        ctl.update_macro(macro, update_body=True)


@admin.register(models.ScenarioTrackerMacro)
class ScenarioTrackerMacroAdmin(admin.ModelAdmin):

    date_hierarchy = 'created'

    search_fields = (
        '=id',
        '=tracker_id',
        '=scenario__slug',
        'name',
    )

    list_display = (
        'id',
        'scenario',
        'name',
        'tracker_id',
        'tracker_queue',
        'is_compatible',
    )

    list_filter = (
        'is_compatible',
        'source',
        ('scenario', EmptyFieldListFilter),
    )

    actions = (
        'sync',
    )

    raw_id_fields = (
        'scenario',
        'tracker_queue',
    )

    readonly_fields = (
        'created',
        'modified',
        'compatibility_error',
        'scenario_creation_url',
    )

    def sync(self, request, queryset):
        sync_macros_task.delay()
        messages.success(request, 'Синхронизация начата')

    def compatibility_error(self, obj):
        url = fetch_ok_iframe_tracker_url(obj.body)
        if not url:
            return None
        if not obj.remote_tracker_macro.macro:
            return str(obj.remote_tracker_macro.exception)

        compatibility_form = ApprovementMacroCompatibilityForm.init(
            url=url,
            tracker_macro=obj.remote_tracker_macro.macro,
            base_initial={
                'scenario': obj.scenario_id,
            },
        )
        return None if compatibility_form.is_valid() else pretty_json(compatibility_form.errors)

    def scenario_creation_url(self, obj):
        if obj.scenario_id:
            return None

        ok_tracker_url = fetch_ok_iframe_tracker_url(obj.body)
        if not (ok_tracker_url and obj.remote_tracker_macro.macro):
            return None

        compatibility_form = ApprovementMacroCompatibilityForm.init(
            url=ok_tracker_url,
            tracker_macro=obj.remote_tracker_macro.macro,
        )
        if not compatibility_form.is_valid():
            return None

        data = compatibility_form.data
        if 'users' in data:
            data.pop('users')
            data['stages'] = compatibility_form.cleaned_data.get('stages', [])

        fields = set(ApprovementForScenarioForm().fields)
        data = {key: value for key, value in data.items() if key in fields}

        json_bytes_string = json.dumps(data, ensure_ascii=False).encode('utf-8')
        encoded_data = base64.urlsafe_b64encode(json_bytes_string).decode('utf-8')

        url = reverse('admin:scenarios_scenario_add')
        url = f'{url}?name={obj.name}&approvement_data={encoded_data}&macro_id={obj.id}'
        return format_url(url, 'new Scenario')


@admin.register(models.ScenarioTrackerTrigger)
class ScenarioTrackerTriggerAdmin(admin.ModelAdmin):

    date_hierarchy = 'created'

    search_fields = (
        '=id',
        '=tracker_id',
        '=scenario__slug',
        'name',
    )

    list_display = (
        'id',
        'scenario',
        'name',
        'tracker_id',
        'tracker_queue',
        'is_active',
        'is_compatible',
    )

    list_filter = (
        'is_active',
        'is_compatible',
        'source',
        ('scenario', EmptyFieldListFilter),
    )

    raw_id_fields = (
        'scenario',
        'tracker_queue',
    )

    readonly_fields = (
        'created',
        'modified',
    )
