from startrek_client.exceptions import Conflict, NotFound

from ok.core.controllers import update_instance, update_list_of_instances
from ok.core.workflow import WorkflowError
from ok.scenarios.choices import SCENARIO_CHANGE_REASONS, SCENARIO_STATUSES
from ok.scenarios.models import Scenario, ScenarioResponsibleGroup, ScenarioTrackerMacro
from ok.staff.tasks import sync_group_memberships_task
from ok.tracker.helpers import get_macro_body, get_macro_iframe_url, replace_url_in_macro_body
from ok.tracker.macros import create_macro, update_macro, delete_macro
from ok.tracker.queues import get_or_create_queue


class ScenarioController:

    def __init__(self, instance):
        self.instance = instance

    @classmethod
    def create(cls, data: dict, initiator):
        data['author'] = initiator
        groups = data.pop('responsible_groups')
        scenario = Scenario.objects.create(**data)
        cls._create_responsible_groups(scenario, groups)
        return scenario

    def update(self, data: dict):
        queue_name = data.pop('tracker_queue', None)
        groups = data.pop('responsible_groups', None)

        update_instance(self.instance, data)
        self._update_responsible_groups(groups)

        macro = self.instance.tracker_macro
        if macro:
            self.update_macro(macro, queue_name)
        elif queue_name:
            self.create_macro(queue_name)

        return self.instance

    def archive(self):
        self.delete_macros()
        self.deactivate_triggers()

        self.instance.status = SCENARIO_STATUSES.archived
        self.instance.save()
        return self.instance

    def restore(self):
        self.restore_macros()
        self.activate_triggers()

        self.instance.status = SCENARIO_STATUSES.active
        self.instance.save()
        return self.instance

    @classmethod
    def _create_responsible_groups(cls, instance, groups):
        responsible_groups = [
            ScenarioResponsibleGroup(scenario=instance, group=group)
            for group in groups
        ]
        ScenarioResponsibleGroup.objects.bulk_create(responsible_groups)
        group_urls = [group.url for group in groups]
        if group_urls:
            sync_group_memberships_task.delay(group_urls)

    def _update_responsible_groups(self, groups):
        if groups is None:
            return
        responsible_groups_data = [
            {
                'scenario_id': self.instance.slug,
                'group_id': group.url,
            }
            for group in groups
        ]
        created, updated, deleted = update_list_of_instances(
            model=ScenarioResponsibleGroup,
            queryset=self.instance.scenario_groups.all(),
            data=responsible_groups_data,
            identifier=('scenario_id', 'group_id'),
        )
        if created:
            group_urls = [group.url for group in groups]
            sync_group_memberships_task.delay(group_urls)
        if any([created, updated, deleted]):
            self.instance._change_reason = SCENARIO_CHANGE_REASONS.responsible_groups_changed
            self.instance.save(update_fields=['modified'])

    # FIXME: здесь немного смешалось создание и обновление
    #  – нужно подумать над тем, как это лучше исправить
    def create_macro(self, queue_name, macro=None):
        tracker_queue = get_or_create_queue(queue_name)
        name = self.instance.name
        body = get_macro_body({'scenario': self.instance.slug})
        try:
            tracker_macro = create_macro(
                queue=queue_name,
                name=name,
                body=body,
            )
        except Conflict:
            # Если в очереди уже есть макрос с таким названием
            raise WorkflowError('macro_already_exists')
        macro = macro or ScenarioTrackerMacro()
        macro.scenario = self.instance
        macro.name = name
        macro.body = body
        macro.tracker_id = tracker_macro.id
        macro.tracker_queue = tracker_queue
        macro.save()
        return macro

    def update_macro(self, macro, queue_name=None, update_body=False):
        name = self.instance.name

        if queue_name and macro.tracker_queue.name != queue_name:
            # Если поменялась очередь
            try:
                delete_macro(
                    key=macro.tracker_id,
                    queue=macro.tracker_queue.name,
                )
            except NotFound:
                # Если в очереди удалили макрос
                pass
            macro = self.create_macro(queue_name, macro)
        elif name != macro.name or update_body:
            params = {}
            if name != macro.name:
                params['name'] = name
            if update_body:
                url = get_macro_iframe_url({'scenario': self.instance.slug})
                body = replace_url_in_macro_body(macro.remote_tracker_macro.macro.body, url)
                macro.body = body
                params['body'] = body
            try:
                update_macro(
                    key=macro.tracker_id,
                    queue=macro.tracker_queue.name,
                    **params,
                )
                macro.name = name
                macro.save()
            except NotFound:
                # Если в очереди удалили макрос
                macro = self.create_macro(macro.tracker_queue.name, macro)
            except Conflict:
                # Если в очереди уже есть макрос с таким названием
                # (пока это не работает, Трекер здесь пятисотит. Но в проде всё ок)
                raise WorkflowError('macro_already_exists')
        return macro

    def delete_macros(self):
        for macro in self.instance.tracker_macros.all():
            try:
                delete_macro(
                    key=macro.tracker_id,
                    queue=macro.tracker_queue.name,
                )
            except NotFound:
                # Если в очереди уже удалили макрос
                pass
        self.instance.tracker_macros.update(is_active=False)

    def restore_macros(self):
        for macro in self.instance.tracker_macros.all():
            self.create_macro(macro.tracker_queue.name, macro)
        self.instance.tracker_macros.update(is_active=True)

    def deactivate_triggers(self):
        # TODO OK-1459
        pass

    def activate_triggers(self):
        # TODO OK-1459
        pass
