from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils.functional import cached_property
from model_utils.models import TimeStampedModel
from simple_history.models import HistoricalRecords

from ok.scenarios.choices import (
    SCENARIO_STATUSES,
    SCENARIO_TRACKER_MACRO_SOURCES,
    SCENARIO_TRACKER_TRIGGER_SOURCES,
)
from ok.tracker.macros import RemoteTrackerMacro


class Scenario(TimeStampedModel):

    slug = models.CharField(max_length=255, primary_key=True)
    status = models.CharField(
        max_length=32,
        choices=SCENARIO_STATUSES,
        default=SCENARIO_STATUSES.active,
    )
    name = models.CharField(max_length=255)
    author = models.CharField(max_length=255)
    approvement_data = JSONField(blank=True, default=dict)
    flow = models.ForeignKey(to='flows.Flow', null=True, blank=True, on_delete=models.SET_NULL)

    # Note: с одной стороны хочется назвать `groups` – консистентно с согласованиями,
    # а с другой хочется назвать более корректно и внятно.
    # Лучше со временем изменить это в согласованиях
    responsible_groups = models.ManyToManyField(
        to='staff.Group',
        through='scenarios.ScenarioResponsibleGroup',
        related_name='+',
        blank=True,
    )

    history = HistoricalRecords()

    @cached_property
    def tracker_macro(self):
        macros = self.tracker_macros.all()
        return macros[0] if macros else None

    def __str__(self):
        return f'Scenario {self.slug}'


class ScenarioTrackerMacro(TimeStampedModel):

    scenario = models.ForeignKey(
        to=Scenario,
        null=True,
        on_delete=models.SET_NULL,
        related_name='tracker_macros',
    )
    name = models.CharField(max_length=255)
    body = models.TextField(null=True)
    tracker_id = models.IntegerField()
    tracker_queue = models.ForeignKey(
        'tracker.Queue',
        on_delete=models.CASCADE,
        related_name='scenario_macros',
    )
    source = models.CharField(
        max_length=32,
        choices=SCENARIO_TRACKER_MACRO_SOURCES,
        default=SCENARIO_TRACKER_MACRO_SOURCES.ok,
        null=True,
        help_text='Откуда появился макрос (через какую систему создан)',
    )
    is_active = models.BooleanField(default=True, null=True)
    is_compatible = models.BooleanField(
        default=True,
        null=True,
        help_text='Совместим ли на данный момент макрос со сценариями в ОКе',
    )

    @cached_property
    def remote_tracker_macro(self):
        return RemoteTrackerMacro(tracker_id=self.tracker_id, queue_name=self.tracker_queue.name)

    def __str__(self):
        return f'ScenarioTrackerMacro ({self.id}), Queue ({self.tracker_queue})'


class ScenarioTrackerTrigger(TimeStampedModel):

    scenario = models.ForeignKey(
        to=Scenario,
        null=True,
        on_delete=models.SET_NULL,
        related_name='tracker_triggers',
    )
    name = models.CharField(max_length=255)
    tracker_id = models.IntegerField()
    tracker_queue = models.ForeignKey(
        'tracker.Queue',
        on_delete=models.CASCADE,
        related_name='scenario_triggers',
    )
    order = models.CharField(max_length=64)
    actions = JSONField(default=list)
    conditions = JSONField(default=list)
    version = models.IntegerField()
    source = models.CharField(
        max_length=32,
        choices=SCENARIO_TRACKER_TRIGGER_SOURCES,
        default=SCENARIO_TRACKER_TRIGGER_SOURCES.ok,
        help_text='Откуда появился триггер (через какую систему создан)',
    )
    is_active = models.BooleanField(default=True)
    is_compatible = models.BooleanField(
        default=True,
        help_text='Совместим ли на данный момент триггер со сценариями в ОКе',
    )

    def __str__(self):
        return f'ScenarioTrackerTrigger ({self.id}), Queue ({self.tracker_queue})'


class ScenarioResponsibleGroup(models.Model):

    scenario = models.ForeignKey(
        to=Scenario,
        on_delete=models.CASCADE,
        related_name='scenario_groups',
    )
    group = models.ForeignKey(
        to='staff.Group',
        on_delete=models.PROTECT,
        related_name='scenario_groups',
    )

    history = HistoricalRecords()

    def __str__(self):
        return f'Scenario ({self.scenario_id}), Group ({self.group_id})'

    class Meta:
        unique_together = ('scenario', 'group')
