import logging

from django.utils.functional import cached_property
from startrek_client.exceptions import NotFound, Forbidden

from ok.api.scenarios.forms import ApprovementMacroCompatibilityForm
from ok.core.sync import SynchronizerBase
from ok.scenarios.choices import SCENARIO_TRACKER_MACRO_SOURCES
from ok.scenarios.models import ScenarioTrackerMacro
from ok.tracker.helpers import fetch_ok_iframe_tracker_url
from ok.tracker.macros import macros
from ok.tracker.models import Queue
from ok.utils.attrs import get_attribute


logger = logging.getLogger(__name__)


class ScenarioTrackerMacroSynchronizer(SynchronizerBase):

    model_class = ScenarioTrackerMacro
    fields_map = {
        'tracker_id': 'id',
        'name': 'name',
        'body': 'body',
        # Это невозможно получить из Трекера – подставляем сами при итерации
        'tracker_queue': 'tracker_queue',
        'is_compatible': '',
    }

    def normalize_remote_item(self, remote_item) -> dict:
        normalized = super().normalize_remote_item(remote_item)
        normalized['name'] = normalized['name'][:255]
        return normalized

    @cached_property
    def existing_macros_map(self):
        queryset = self.model_class.objects.select_related('tracker_queue')
        return {(m.tracker_queue.name, m.tracker_id): m for m in queryset}

    def iter(self):
        for queue in Queue.objects.all():
            for macro in self._get_macros(queue.name):
                macro.tracker_queue = queue
                yield macro

    def sync(self):
        to_create = []
        to_update = []
        to_delete_ids = []
        to_restore_ids = []

        logger.info('Macros sync started')

        for tracker_macro in self.iter():
            ok_url = fetch_ok_iframe_tracker_url(tracker_macro.body)
            if not ok_url:
                continue
            macro_data = self.normalize_remote_item(tracker_macro)
            key = (macro_data['tracker_queue'].name, macro_data['tracker_id'])
            macro = self.existing_macros_map.pop(key, None)

            compatibility_form = ApprovementMacroCompatibilityForm.init(
                url=ok_url,
                tracker_macro=tracker_macro,
                base_initial={
                    'scenario': get_attribute(macro, 'scenario_id'),
                },
            )
            macro_data['is_compatible'] = compatibility_form.is_valid()

            if macro:
                if not self.has_anything_changed(macro, macro_data):
                    continue
                if macro.scenario_id is not None:
                    logger.warning('Macro linked to scenario was modified in Tracker: %d', macro.id)
                    to_restore_ids.append(macro.id)
                    continue
                for field_name in self.fields_map:
                    setattr(macro, field_name, macro_data[field_name])
                to_update.append(macro)
            else:
                to_create.append(self.model_class(
                    source=SCENARIO_TRACKER_MACRO_SOURCES.tracker,
                    **macro_data,
                ))

        self.model_class.objects.bulk_create(to_create, batch_size=1000)
        self.model_class.objects.bulk_update(to_update, self.fields_map.keys(), batch_size=1000)

        for macro in self.existing_macros_map.values():
            if macro.scenario_id:
                logger.warning('Macro linked to scenario was deleted from Tracker: %d', macro.id)
                to_restore_ids.append(macro.id)
            else:
                to_delete_ids.append(macro.id)

        self.model_class.objects.filter(id__in=to_delete_ids).delete()

        # TODO OK-1222: Решить, что делать, если макросы, привязанные к сценарию,
        #  изменили или удалили через Трекер (особенно, когда source == 'ok')
        logger.info('Macros to restore: %s', to_restore_ids)
        to_restore_ids = []

        logger.info(
            'Macros sync completed: %d created, %d updated, %d deleted, %d restored',
            len(to_create),
            len(to_update),
            len(to_delete_ids),
            len(to_restore_ids),
        )

    @staticmethod
    def _get_macros(queue_name):
        try:
            return macros.get_all(queue=queue_name)
        except (NotFound, Forbidden) as exc:
            logger.warning('Could not get macros for queue %s', queue_name, exc_info=exc)
        return []


def sync_macros():
    ScenarioTrackerMacroSynchronizer().sync()
