from django.utils.functional import cached_property

from ok.core.workflow import Action, Workflow
from ok.scenarios.choices import SCENARIO_STATUSES
from ok.scenarios.controllers import ScenarioController


class ScenarioBaseAction(Action):

    @cached_property
    def ctl(self):
        return ScenarioController(self.instance)

    def has_permission(self):
        return self.wf.is_responsible


class UpdateAction(ScenarioBaseAction):

    valid_statuses = (SCENARIO_STATUSES.active,)

    def perform(self, **params):
        return self.ctl.update(params)


class ArchiveAction(ScenarioBaseAction):

    valid_statuses = (SCENARIO_STATUSES.active,)

    def perform(self):
        return self.ctl.archive()


class RestoreAction(ScenarioBaseAction):

    valid_statuses = (SCENARIO_STATUSES.archived,)

    def perform(self):
        return self.ctl.restore()


class ScenarioWorkflow(Workflow):

    @cached_property
    def is_author(self):
        return self.instance.author == self.user

    @cached_property
    def is_responsible(self):
        return self.is_author or self.user in self.group_members

    @cached_property
    def group_members(self):
        members = set()
        for group in self.instance.responsible_groups.all():
            for membership in group.memberships.all():
                members.add(membership.login)
        return members

    def get_actions(self):
        actions = super().get_actions()
        actions.setdefault('clone', True)
        return actions

    ACTION_MAP = {
        'update': UpdateAction,
        'archive': ArchiveAction,
        'restore': RestoreAction,
    }
