import logging

from django.contrib.auth import get_user_model

from ok.approvements.models import ApprovementGroup
from ok.core.choices import LANGUAGES
from ok.core.sync import SynchronizerBase
from ok.scenarios.models import ScenarioResponsibleGroup
from ok.staff.models import Group, GroupMembership
from ok.utils.lists import chunks
from ok.utils.staff import get_staff_repo, get_staff_group_memberships


User = get_user_model()

logger = logging.getLogger(__name__)


class StaffSynchronizerBase(SynchronizerBase):

    resource_type = None
    id_field = None

    def __init__(self, id_list=None):
        self.repo = get_staff_repo(self.resource_type)
        self.id_list = id_list

    def get_params(self):
        params = {
            '_limit': 1000,
            '_fields': ','.join(self.fields_map.values()),
            '_sort': 'id',
        }
        if self.id_list is not None:
            params[self.fields_map[self.id_field]] = ','.join(self.id_list)
        return params

    def iter(self):
        return self.repo.getiter(self.get_params())

    def get_objects_mapping(self):
        return self.model_class.objects.in_bulk(id_list=self.id_list, field_name=self.id_field)

    def get_mapping_key(self, normalized_item):
        return normalized_item[self.id_field]

    def sync(self):
        data = self.iter()
        objects_mapping = self.get_objects_mapping()
        to_create = []
        to_update = []

        logger.info('%s sync started: %d to sync', self.model_class.__name__, data.total)

        for item in data:
            normalized_item = self.normalize_remote_item(item)
            db_obj = objects_mapping.pop(self.get_mapping_key(normalized_item), None)
            is_creation = not db_obj
            if is_creation:
                db_obj = self.model_class()

            if self.has_anything_changed(db_obj, normalized_item):
                for field_name in self.fields_map:
                    setattr(db_obj, field_name, normalized_item[field_name])
                if is_creation:
                    to_create.append(db_obj)
                else:
                    to_update.append(db_obj)

        self.model_class.objects.bulk_create(to_create, batch_size=1000)
        update_fields = set(self.fields_map)
        pk_name = self.model_class._meta.pk.name
        update_fields.discard(pk_name)
        self.model_class.objects.bulk_update(to_update, update_fields, batch_size=1000)
        logger.info(
            '%s sync completed: %d created, %d updated',
            self.model_class.__name__,
            len(to_create),
            len(to_update),
        )


class UserSynchronizer(StaffSynchronizerBase):

    model_class = User
    resource_type = 'person'
    fields_map = {
        'username': 'login',
        'staff_id': 'id',
        'uid': 'uid',
        'email': 'work_email',
        'first_name': 'name.first.ru',
        'last_name': 'name.last.ru',
        'first_name_en': 'name.first.en',
        'last_name_en': 'name.last.en',
        'is_dismissed': 'official.is_dismissed',
        'language': 'language.ui',
        'affiliation': 'official.affiliation',
    }
    id_field = 'username'

    def normalize_remote_item(self, remote_item):
        normalized = super().normalize_remote_item(remote_item)
        normalized['is_dismissed'] = normalized['is_dismissed'] or False
        normalized['language'] = normalized['language'] or LANGUAGES.ru
        return normalized


class GroupSynchronizer(StaffSynchronizerBase):

    model_class = Group
    resource_type = 'group'
    fields_map = {
        'url': 'url',
        'staff_id': 'id',
        'is_deleted': 'is_deleted',
        'name_ru': 'name',
        'name_en': 'department.name.full.en',
        'type': 'type',
    }
    id_field = 'url'

    def get_params(self):
        params = super().get_params()
        params['is_deleted'] = 'true,false'
        return params

    def normalize_remote_item(self, remote_item):
        normalized = super().normalize_remote_item(remote_item)
        normalized['is_deleted'] = normalized['is_deleted'] or False
        normalized['name_en'] = normalized['name_en'] or normalized['name_ru']
        return normalized


def sync_users():
    UserSynchronizer().sync()


def sync_groups(urls=None):
    GroupSynchronizer(urls).sync()


def _get_active_responsible_groups(model_class):
    return (
        model_class.objects
        .filter(
            group__staff_id__isnull=False,
            group__is_deleted=False,
        )
        .values_list('group_id', flat=True)
    )


def sync_group_memberships(group_urls=None, chunk_size=50):
    scenario_responsibles_qs = _get_active_responsible_groups(ScenarioResponsibleGroup)
    approvement_responsibles_qs = _get_active_responsible_groups(ApprovementGroup)
    active_group_urls = set(approvement_responsibles_qs.union(scenario_responsibles_qs))
    if group_urls is None:
        group_urls = active_group_urls
        not_active_groups = Group.objects.exclude(url__in=active_group_urls)
    else:
        group_urls = set(group_urls)
        not_active_groups = group_urls - active_group_urls
        group_urls &= active_group_urls
    for group_urls_chunk in chunks(group_urls, chunk_size):
        memberships = GroupMembership.objects.filter(group__in=group_urls_chunk)
        memberships_map = {(m.group_id, m.login): m.id for m in memberships}
        to_create = []
        for membership_data in get_staff_group_memberships(group_urls_chunk):
            key = group_url, login = membership_data['group_url'], membership_data['login']
            if key in memberships_map:
                memberships_map.pop(key)
            else:
                to_create.append(GroupMembership(group_id=group_url, login=login))

        GroupMembership.objects.bulk_create(to_create, batch_size=1000)
        GroupMembership.objects.filter(id__in=memberships_map.values()).delete()
    GroupMembership.objects.filter(group__in=not_active_groups).delete()
