from django.contrib.auth.models import AbstractUser
from django.db import models
from django.utils.translation import get_language

from ok.core.choices import LANGUAGES


class User(AbstractUser):

    staff_id = models.IntegerField(unique=True, null=True)
    uid = models.CharField(max_length=32, unique=True, null=True)
    first_name = models.CharField(max_length=255)
    last_name = models.CharField(max_length=255)
    first_name_en = models.CharField(max_length=255, null=True)
    last_name_en = models.CharField(max_length=255, null=True)
    language = models.CharField(
        max_length=2,
        choices=LANGUAGES,
        default=LANGUAGES.ru,
        null=True,
    )
    affiliation = models.CharField(max_length=32, null=True)
    is_dismissed = models.BooleanField(default=False, null=True)

    def _get_localized_attr(self, attr, default=None):
        # Используем сортированный словарь для приоритизации языков
        # – наиболее высокий приоритет у последнего
        languages = dict.fromkeys((LANGUAGES.en, LANGUAGES.ru))
        lang = get_language()[:2]
        lang = lang if lang in languages else LANGUAGES.ru
        languages[lang] = languages.pop(lang)
        values = (getattr(self, f'{attr}_{lang}') for lang in reversed(languages))
        return next((value for value in values if value), default)

    @property
    def first_name_ru(self):
        return self.first_name

    @property
    def last_name_ru(self):
        return self.last_name

    @property
    def localized_first_name(self):
        return self._get_localized_attr('first_name', '')

    @property
    def localized_last_name(self):
        return self._get_localized_attr('last_name', '')

    @property
    def localized_full_name(self):
        return f'{self.localized_first_name} {self.localized_last_name}'.strip()

    class Meta:
        db_table = 'auth_user'
