from threading import local

from ok.core.permissions import is_yauser_authenticated
from ok.utils.attrs import get_attribute
from ok.utils.tvm import get_user_ticket


class _RequestContext(object):
    """
    Псевдо-синглтон для сохранения какого-либа контекста в рамках http-запроса.
    """
    context_fields = ['request', 'flow_name']

    def __init__(self):
        self._data = local()
        self.reset()

    def set(self, key, value):
        setattr(self._data, key, value)

    def init(self, **kwargs):
        for key, value in kwargs.items():
            self.set(key, value)

    def reset(self):
        reset_data = dict.fromkeys(self.context_fields)
        self.init(**reset_data)

    @property
    def user(self):
        if self.request and is_yauser_authenticated(self.request.yauser):
            return self.request.yauser.login
        return None

    @property
    def user_ticket(self):
        assert self.request is not None
        if not hasattr(self.request, '_user_ticket'):
            self.request._user_ticket = get_user_ticket(self.request)
        return self.request._user_ticket

    @property
    def referer(self):
        return get_attribute(self.request, 'headers.referer')

    @property
    def ok_session_id(self):
        return get_attribute(self.request, 'headers.x-ok-session-id')

    def __getattr__(self, item):
        return getattr(self._data, item, None)


request_context = _RequestContext()
