import logging

import requests

from django.conf import settings
from ids.registry import registry

from ok.utils.http_session import requests_retry_session
from ok.utils.lists import chunks
from ok.utils.tvm import get_service_ticket, get_user_ticket


logger = logging.getLogger(__name__)


def get_staff_repo(resource_type):
    repo = registry.get_repository(
        service='staff',
        resource_type=resource_type,
        user_agent='ok',
        oauth_token=settings.OK_ROBOT_TOKEN,
        timeout=settings.STAFF_API_TIMEOUT,
        retries=settings.STAFF_API_RETRIES,
    )
    return repo


def get_staff_iter(resource_type, params):
    repo = get_staff_repo(resource_type)
    params.setdefault('_limit', 1000)
    params.setdefault('_sort', 'id')
    return repo.getiter(params)


def get_staff_iter_nopage(resource_type, params, min_id=0):
    repo = get_staff_repo(resource_type)

    assert '_query' not in params
    params.setdefault('_limit', 1000)
    params['_sort'] = 'id'
    fields = params.get('_fields')
    if fields:
        params['_fields'] = fields + ',id'

    while True:
        params['_query'] = f'id > {min_id}'
        result = repo.get_nopage(params)
        if not result:
            break
        min_id = result[-1]['id']
        yield from result


def get_staff_users_iter(params):
    return get_staff_iter(
        resource_type='person',
        params=params,
    )


def get_staff_users_by_logins(login_list, params):
    for login_chunk in chunks(login_list):
        params['login'] = ','.join(login_chunk)
        yield from get_staff_users_iter(params)


def get_staff_groupmembership_iter(params):
    return get_staff_iter_nopage(
        resource_type='groupmembership',
        params=params,
    )


def get_staff_group_memberships(group_urls):
    memberships = get_staff_groupmembership_iter({
        'group.url': ','.join(group_urls),
        'person.official.is_dismissed': False,
        '_fields': 'group.url,person.login',
    })
    for membership in memberships:
        yield {
            'group_url': membership['group']['url'],
            'login': membership['person']['login'],
        }


class GapNewhireCountsApiError(Exception):
    pass


def get_gap_newhire_data(request):
    session = requests_retry_session()
    try:
        response = session.get(
            url=f'{settings.STAFF_PROFILE_API}gap_newhire_counts/',
            headers={
                'X-Ya-Service-Ticket': get_service_ticket(settings.TVM_STAFF_CLIENT_ID),
                'X-Ya-User-Ticket': get_user_ticket(request),
            },
            timeout=settings.STAFF_API_TIMEOUT,
        )
    except requests.exceptions.RequestException:
        logger.error('Gap-newhire-counts API is not responding')
        raise GapNewhireCountsApiError('Gap-newhire-counts API is not responding')

    if not response.ok:
        logger.error(
            'Gap-newhire-counts API responded with status %s: %s',
            response.status_code,
            response.content.decode('utf-8'),
        )
        raise GapNewhireCountsApiError(
            f'Gap-newhire-counts API wrong status {response.status_code}',
        )

    return response.json()
