import logging

from django.conf import settings
from ids.connector import http, plugins_lib
from ids.exceptions import BackendError

from ok.utils.tvm import get_service_ticket


logger = logging.getLogger(__name__)


class XivaInternalError(Exception):
    pass


class XivaInternalConnector(http.HttpConnector):
    """
    Отправка уведомлений в корпоративное окружение xiva
    https://console.push.yandex-team.ru/#overview
    """
    url_prefix = '/v2'
    url_patterns = {
        'batch_send': '/batch_send',
    }

    plugins = [
        plugins_lib.TVM2ServiceTicket,
        plugins_lib.JsonResponse,
    ]

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('user_agent', 'ok')
        kwargs.setdefault(
            'service_ticket',
            get_service_ticket(settings.TVM_XIVA_INTERNAL_CLIENT_ID),
        )
        super().__init__(*args, **kwargs)

    @property
    def config(self):
        return {
            'protocol': 'https',
            'host': 'push.yandex-team.ru',
        }

    def _push_notification(self, service, event, user_uids, tags=None, payload=None, repack=None):
        params = {
            'service': service,
            'event': event,
        }
        data = {
            'tags': tags or [],
            'payload': payload or {},
            'recipients': user_uids,
            'repack': repack or {},
        }
        return self.post(
            resource='batch_send',
            params=params,
            json=data,
        )

    def parse_result(self, uid, result):
        """
        Метод возвращает словарь с данными по неуспешным отправкам пушей.
        Если у пользователя несколько подписок, то в result придёт список.
        Ретраить будем в этом случае только для тех подписок, для которых была неуспешная отправка.
        """
        if isinstance(uid, dict):
            uid = list(uid.items())[0]
        if isinstance(result, dict):
            if result['code'] >= 400:
                return {uid: result}
            return {}
        failed_results_by_subscription = {}
        for subscription_result in result:
            if subscription_result['code'] >= 400:
                key = (uid, subscription_result['id'])
                failed_results_by_subscription[key] = subscription_result
        return failed_results_by_subscription

    def push_batch_notification(self, service: str, event: str, user_uids: list, tags: list = None,
                                payload: dict = None, repack: dict = None):
        """
        Отправить нотификацию нескольким пользователям.
        Одному пользователю отправляем пуши тоже через batch, т.к. при наличии у него нескольких
        подписок нужен ретрай батчем по списку подписок с неуспешной отправкой
        Документация: https://console.push.yandex-team.ru/#api-reference-batch-send

        :param service: имя сервиса, зарегистрированного в xiva
        :param event: короткий заголовок уведомления, полезен для статистики и фильтрации
        :param user_uids: список uid пользователей или словарей вида {<uid>: <id подписки>}
        :param tags: теги для простой фильтрации уведомлений
        :param payload: тело уведомления
        :param repack: правила переупаковки данных и сами данные, отправляемые на разные платформы
        """
        retries = 3

        for attempt in range(retries):
            failed_results = {}
            try:
                response = self._push_notification(service, event, user_uids, tags, payload, repack)
                for uid, result in zip(user_uids, response['results']):
                    failed = self.parse_result(uid, result)
                    failed_results.update(failed)
                user_uids = [
                    {result[0]: result[1]} if isinstance(result, tuple) else result
                    for result in failed_results
                ]
                if user_uids:
                    raise BackendError(failed_results)
                return response
            except BackendError as exc:
                is_final_attempt = attempt + 1 == retries
                level = logging.ERROR if is_final_attempt else logging.WARNING
                logger.log(
                    level,
                    'Could not send notification to users (%d)',
                    attempt,
                    exc_info=exc,
                )
                if is_final_attempt:
                    raise XivaInternalError(exc)


def push_notification_to_app(event, user_uids, title, description='', url=settings.OK_URL,
                             data=None):
    connector = XivaInternalConnector()
    data = data or {}
    data['url'] = url
    payload = {
        'service': 'ok',
        'notification': {
            'title': title,
            'description': description,
            'data': data,
        },
    }
    repack = {
        'apns': {  # ios
            'aps': {
                'alert': {
                    'title': 'OK',
                    'subtitle': title,
                    'body': description,
                },
                'thread-id': 'service-ok',  # группировка пушей
                'mutable-content': 1,
            },
        },
        'fcm': {  # android
            'notification': {
                'title': title,
                'body': description,
            },
        },
    }
    return connector.push_batch_notification(
        service='yandex-team-app',
        event=event,
        user_uids=user_uids,
        tags=['ok'],
        payload=payload,
        repack=repack,
    )
