from enum import Enum
from typing import Dict, List


def get_dev_mode():
    try:
        import os
    except ImportError:
        return False
    return bool(int(os.getenv('DEV_MODE', 0)))


DEV_MODE = get_dev_mode()
if DEV_MODE:
    from .flow_mocks import *


class FlowEnum(Enum):
    @classmethod
    def choices(cls):
        return [v.value for v in cls]


class TopDepartments(FlowEnum):
    YANDEX = 'yandex'
    YANDEX_KPB = 'outstaff'
    TAXI = 'yandex_rkub_taxi_dep80006'
    TAXI_KPB = 'outstaff_2289'
    DRIVE = 'yandex_content_8006'
    SEARCH_ADV_CLOUD = 'yandex_dep79124'
    SEARCH_OUTSTAFF = 'outstaff_3210'
    S_OUT_DRIVE = 'as_7861_dep79112'
    S_OUT_MEDIA = 'as_7861_dep61649'
    CLOUD = 'yandex_exp_9053'
    SEARCH_PORTAL = 'yandex_main_searchadv'
    UMII = 'yandex_main_searchadv_ai'
    ZEN = 'yandex_rkub_discovery'
    ZEN_KPB = 'outstaff_9036'


class TransferTypes(FlowEnum):
    """
        Типы переводов между подразделениями и БП
    """
    CHANGE_BRANCH = 'change_branch'
    FROM_INTERN = 'from_intern'
    FROM_MATERNITY = 'from_maternity'
    WITH_BUDGET = 'with_budget'
    WO_BUDGET = 'wo_budget'


class Case:
    def is_equal(self, case_type):
        return self.one_of([case_type])

    def one_of(self, case_types):
        return isinstance(self, tuple(case_types))

    def __init__(self, person_changes: 'PersonChanges', department: flow_api.Department):
        self.person_changes = person_changes
        self.department = department


class TransferChangeBranchAccept(Case):
    pass


class TransferChangeBranchGive(Case):
    pass


class TransferFromInternAccept(Case):
    pass


class TransferFromInternGive(Case):
    pass


class TransferFromMaternityAccept(Case):
    pass


class TransferFromMaternityGive(Case):
    pass


class TransferWithBudgetGive(Case):
    pass


class TransferWithBudgetAccept(Case):
    pass


class TransferWoBudgetGive(Case):
    pass


class TransferWoBudgetAccept(Case):
    pass


class TransferChangeBgWoBudgetGive(Case):
    pass


class TransferChangeBgWoBudgetAccept(Case):
    pass


class TransferChangeBgWithBudgetGive(Case):
    pass


class TransferChangeBgWithBudgetAccept(Case):
    pass


class ChangeOrg(Case):
    pass


class ChangeSalary(Case):
    pass


class ChangePosition(Case):
    pass


class ChangeOffice(Case):
    pass


class ChangeRate(Case):
    pass


class Cases(FlowEnum):
    TRANSFER_CHANGE_BRANCH_ACCEPT = TransferChangeBranchAccept
    TRANSFER_CHANGE_BRANCH_GIVE = TransferChangeBranchGive

    TRANSFER_FROM_INTERN_ACCEPT = TransferFromInternAccept
    TRANSFER_FROM_INTERN_GIVE = TransferFromInternGive

    TRANSFER_FROM_MATERNITY_ACCEPT = TransferFromMaternityAccept
    TRANSFER_FROM_MATERNITY_GIVE = TransferFromMaternityGive

    TRANSFER_WITH_BUDGET_GIVE = TransferWithBudgetGive
    TRANSFER_WITH_BUDGET_ACCEPT = TransferWithBudgetAccept
    TRANSFER_WO_BUDGET_GIVE = TransferWoBudgetGive
    TRANSFER_WO_BUDGET_ACCEPT = TransferWoBudgetAccept

    TRANSFER_CHANGE_BG_WO_BUDGET_GIVE = TransferChangeBgWoBudgetGive
    TRANSFER_CHANGE_BG_WO_BUDGET_ACCEPT = TransferChangeBgWoBudgetAccept
    TRANSFER_CHANGE_BG_WITH_BUDGET_GIVE = TransferChangeBgWithBudgetGive
    TRANSFER_CHANGE_BG_WITH_BUDGET_ACCEPT = TransferChangeBgWithBudgetAccept

    CHANGE_ORG = ChangeOrg
    CHANGE_SALARY = ChangeSalary
    CHANGE_POSITION = ChangePosition
    CHANGE_OFFICE = ChangeOffice
    CHANGE_RATE = ChangeRate


OVERLAY_CASES_VALUES = {
    TransferChangeBranchAccept,
    TransferChangeBranchGive,
    TransferFromInternAccept,
    TransferFromInternGive,
    TransferFromMaternityAccept,
    TransferFromMaternityGive,
    TransferWoBudgetGive,
    TransferWoBudgetAccept,
    TransferChangeBgWoBudgetGive,
    TransferChangeBgWoBudgetAccept,
    TransferChangeBgWithBudgetGive,
    TransferChangeBgWithBudgetAccept,
    TransferWithBudgetGive,
    TransferWithBudgetAccept,
}


class FakeDepartmentException(Exception):
    pass


class PersonChanges:

    def __init__(self, context: dict, person: flow_api.Person):
        self._context = context
        self.person: flow_api.Person = person

    @property
    def is_changing_pay_system(self) -> bool:
        """Смена системы оплаты труда (Заявка)"""
        salary = self._context.get('salary', {})
        return salary.get('old_wage_system') != salary.get('new_wage_system')

    @property
    def is_changing_salary(self) -> bool:
        """Меняется оклад =  Новая ЗП/Новая ставка != Старая ЗП/Старая ставка (Заявка)"""
        salary = self._context.get('salary')
        if not salary:
            return False
        old_currency = salary.get('old_currency')
        new_currency = salary.get('new_currency')
        if old_currency != new_currency:
            return True

        old_sum = float(salary.get('old_salary', 0))
        new_sum = float(salary.get('new_salary', 0))
        old_rate = float(salary.get('old_rate', 1))
        new_rate = float(salary.get('new_rate', 1))

        return new_sum / new_rate != old_sum / old_rate

    @property
    def is_changing_only_rate(self) -> bool:
        """Меняется оклад =  Новая ЗП/Новая ставка != Старая ЗП/Старая ставка (Заявка)"""
        salary = self._context.get('salary')
        if not salary:
            return False

        old_currency = salary.get('old_currency')
        new_currency = salary.get('new_currency')
        if old_currency != new_currency:
            return False

        old_sum = float(salary.get('old_salary', 0))
        new_sum = float(salary.get('new_salary', 0))
        old_rate = float(salary.get('old_rate', 1))
        new_rate = float(salary.get('new_rate', 1))

        return new_sum / new_rate == old_sum / old_rate

    @property
    def is_tab_salary(self) -> bool:
        """Вкладка ЗП"""
        return 'salary' in self._context

    @property
    def is_salary_piecework(self) -> bool:
        """Изменение ЗП у сдельщика?"""
        salary = self._context.get('salary')
        old_wage_system = salary.get('old_wage_system')
        new_wage_system = salary.get('new_wage_system')

        return (old_wage_system == 'piecework') & (new_wage_system == 'piecework')

    @property
    def is_tab_office(self) -> bool:
        """Вкладка Офис"""
        return 'office' in self._context

    @property
    def is_tab_organization(self) -> bool:
        """Вкладка Организация"""
        return 'organization' in self._context

    @property
    def is_tab_position(self) -> bool:
        """Вкладка Должность"""
        return 'position' in self._context

    @property
    def is_changing_grade(self) -> bool:
        """Меняется грейд"""
        return self._context.get('grade', {}).get('new_grade', 0) != 0

    @property
    def is_tab_department(self) -> bool:
        """Вкладка Подразделение"""
        return 'department' in self._context

    @property
    def is_changing_branch(self) -> bool:
        """Происходит ли смена ветки между КПБ и не КПБ"""
        # Не будет работать, если new_department_chain не работает для fake_department
        return self.is_tab_department and (
                (
                        self.person.is_external and
                        flow_api.Department('yandex') in self.new_department_chain
                ) or (
                        not self.person.is_external and
                        flow_api.Department('outstaff') in self.new_department_chain
                )
        )

    @property
    def transfer_type(self) -> TransferTypes or None:
        """Сценарий перевода (Заявка)"""

        if not self.is_tab_department:
            return None

        if self._context.get('department', {}).get('from_maternity_leave'):
            return TransferTypes.FROM_MATERNITY

        if self.person.is_external and self.is_changing_branch:
            return TransferTypes.CHANGE_BRANCH

        if self._context.get('department', {}).get('with_budget'):
            return TransferTypes.WITH_BUDGET

        if self.person.is_intern:
            return TransferTypes.FROM_INTERN

        return TransferTypes.WO_BUDGET

    # --------

    @property
    def new_department(self) -> flow_api.Department:
        # TODO: Департамент может быть ещё не созданым- нужно обработать fake_department
        department_data = self._context.get('department', {})

        if department_data.get('fake_department'):
            raise FakeDepartmentException

        new_department = department_data.get('department')
        if new_department is None:
            return self.person.department
        else:
            return flow_api.Department(new_department)

    @property
    def new_department_chain(self) -> List[flow_api.Department]:
        return self.new_department.get_ancestors()

    @property
    def is_changing_department(self) -> bool:
        return self.person.department != self.new_department

    @property
    def new_top_department(self) -> flow_api.Department:
        return get_top_department(self.new_department_chain)

    # --------

    def get_cases(self) -> List[Case]:
        cases = []
        if self.is_tab_department:
            if self.transfer_type == TransferTypes.FROM_INTERN:
                if self.new_top_department == get_top_department(self.person.department_chain):
                    cases.append(Cases.TRANSFER_FROM_INTERN_ACCEPT.value(self, self.person.department))
                else:
                    cases.append(Cases.TRANSFER_FROM_INTERN_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_FROM_INTERN_ACCEPT.value(self, self.new_department))
            elif self.transfer_type == TransferTypes.CHANGE_BRANCH:
                cases.append(Cases.TRANSFER_CHANGE_BRANCH_GIVE.value(self, self.person.department))
                cases.append(Cases.TRANSFER_CHANGE_BRANCH_ACCEPT.value(self, self.new_department))
            elif self.transfer_type == TransferTypes.FROM_MATERNITY:
                if self.new_top_department == get_top_department(self.person.department_chain):
                    cases.append(Cases.TRANSFER_FROM_MATERNITY_ACCEPT.value(self, self.new_department))
                else:
                    cases.append(Cases.TRANSFER_FROM_MATERNITY_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_FROM_MATERNITY_ACCEPT.value(self, self.new_department))
            elif self.transfer_type == TransferTypes.WITH_BUDGET:
                if self.new_top_department == get_top_department(self.person.department_chain):
                    cases.append(Cases.TRANSFER_WITH_BUDGET_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_WITH_BUDGET_ACCEPT.value(self, self.new_department))
                else:
                    cases.append(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_ACCEPT.value(self, self.new_department))
            elif self.transfer_type == TransferTypes.WO_BUDGET:
                if self.new_top_department == get_top_department(self.person.department_chain):
                    cases.append(Cases.TRANSFER_WO_BUDGET_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_WO_BUDGET_ACCEPT.value(self, self.new_department))
                else:
                    cases.append(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_GIVE.value(self, self.person.department))
                    cases.append(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_ACCEPT.value(self, self.new_department))

        # Хоп, костылек
        a = []
        for c in cases:
            a.append(c.__class__)

        wo_overlay_case = not set(a) & OVERLAY_CASES_VALUES

        if (self.is_changing_salary or self.is_changing_grade) and wo_overlay_case:
            cases.append(Cases.CHANGE_SALARY.value(self, self.person.department))

        if self.is_changing_only_rate and wo_overlay_case:
            cases.append(Cases.CHANGE_RATE.value(self, self.person.department))

        if self.is_tab_office and wo_overlay_case:
            cases.append(Cases.CHANGE_OFFICE.value(self, self.person.department))
        if self.is_tab_position and wo_overlay_case:
            cases.append(Cases.CHANGE_POSITION.value(self, self.person.department))

        if self.is_tab_organization and wo_overlay_case:
            cases.append(Cases.CHANGE_ORG.value(self, self.person.department))

        return cases


def get_top_department(departments: List[flow_api.Department]):
    for dep in departments:
        if dep.url in TopDepartments.choices():
            return dep


class ProposalContext:
    """
    =====Признаки
    - Вкладка Иерархия у подразделения (Заявка) - точно ли в этом классе?
    - Изменение внутри БГ (Заявка) - точно ли в этом классе?
    """

    def __init__(self, context: dict):
        self._context = context

        self._persons_changes = None
        self._departments_changes = None

    @property
    def is_personal(self) -> bool:
        return self._context.get('ticket_type') == 'personal'

    @property
    def is_changing_hierarchy(self):
        """Вкладка Иерархия у подразделения (Заявка)"""
        raise NotImplemented
        return bool(self._context.get(''))
        # changing_hierarchy = self._context.

    def get_departments_changes(self):
        raise NotImplemented

    def get_persons_changes(self) -> Dict[str, PersonChanges]:
        if self._persons_changes is None:
            self._persons_changes = {}
            for person_context in self._context['person_actions']:
                person = flow_api.Person(person_context['login'])
                self._persons_changes[person.login] = PersonChanges(context=person_context, person=person)
        return self._persons_changes

    def get_current_person_changes(self) -> PersonChanges:
        assert self.is_personal, 'Только контекст с персональным тикетом имеет текущего сотрудника'
        return self.get_persons_changes()[self._context.get('ticket_person')]


"""
Нужно написать функции для сбора этих цепочек:
Для Такси зовём всех параллельно
-СтаршийHRBP(РуководительHRBP)+
-НадеждаГущина(login)+
-HRBP(посотрудникуилиподразделению)+
-Руководительструктуры(Даня-2)(посотрудникуилиподразделению)+
-Непосредственныеруководители(посотруднику)+
-Согласованиенетребуется
-Аналитик(посотрудникуилиподразделению)?
-Наименьшийобщийруководитель(дляПП)
-Цепочкаруководителей(дляПП)
"""


class ApproverChains:

    @staticmethod
    def _persons_2_login(persons: List[flow_api.Person]) -> List[str]:
        return [p.login for p in persons]

    @classmethod
    def get_lead_hrbp(cls, case: Case) -> List[str]:
        hrs = case.person_changes.person.hr_partners
        hrbp_heads = hrs[0].heads_chain
        return cls._persons_2_login([hrbp_heads[0]])

    @classmethod
    def get_hr_director(cls, case: Case) -> List[str]:
        return ['nadyag']

    @classmethod
    def get_hrbp(cls, case: Case) -> List[str]:
        hrs = case.department.hr_partners
        first_hrbp = hrs[-1]
        return cls._persons_2_login([first_hrbp])

    @classmethod
    def get_head(cls, case: Case) -> List[str]:
        return cls._persons_2_login([case.department.head])

    @classmethod
    def get_daniilsh_minus2(cls, case: Case) -> List[str]:
        heads_list = case.person_changes.person.heads_chain
        i = heads_list.index('daniilsh')
        return cls._persons_2_login([heads_list[i - 2]])

    @classmethod
    def get_accepting_head(cls, case: Case) -> List[str]:
        new_dep = case.person_changes.new_department
        heads_list = new_dep.heads_chain
        return cls._persons_2_login([heads_list[0]])

    @classmethod
    def get_first_common_head(cls, case: Case) -> List[str]:
        heads_list = case.person_changes.person.heads_chain
        heads_list = cls._persons_2_login(heads_list)
        new_dep = case.person_changes.new_department
        accepting_heads_list = new_dep.heads_chain
        accepting_heads_list = cls._persons_2_login(accepting_heads_list)
        heads = []
        for head in heads_list:
            if head in set(accepting_heads_list):
                heads.append(head)
        return [heads[0]]

    @classmethod
    def first_common_dep_if_level_5(cls, case: Case) -> bool:
        dep_chain = case.person_changes.person.department_chain
        new_dep_chain = case.person_changes.new_department_chain
        common_deps = []
        for dep in dep_chain:
            if dep in set(new_dep_chain):
                common_deps.append(dep)

        if len(common_deps) >= 5:
            return True
        else:
            return False


def main(context):
    proposal_ctx = ProposalContext(context)
    approvers = []

    if not proposal_ctx.is_personal:
        """
        ===== Сценарии
        1. Изменения в структуре
            - Вкладка Иерархия у подразделения
            - Будет развилка по подразделениям внутри БГ (изменение потолков)
            - Не перекрывается ничем
        """
        drop_approvement()
        return

    person_changes = proposal_ctx.get_current_person_changes()
    # Перечисляем кейсы и их соответсвие сценариям
    # Нужно как-то обозначить, в какой ветке мы находимся - как?
    try:
        current_cases = person_changes.get_cases()
    except FakeDepartmentException:
        drop_approvement()
        return

    taxi_kpb_cases: List[Case] = [
        case for case in current_cases
        if get_top_department(case.department.get_ancestors()).url == TopDepartments.TAXI_KPB.value
    ]
    search_out_cases = [
        case for case in current_cases
        if get_top_department(case.department.get_ancestors()).url == TopDepartments.SEARCH_OUTSTAFF.value
    ]
    zen_kpb_cases = [
        case for case in current_cases
        if get_top_department(case.department.get_ancestors()).url == TopDepartments.ZEN_KPB.value
    ]
    search_out_drive_cases = [
        case for case in current_cases
        if get_top_department(case.department.get_ancestors()).url == TopDepartments.S_OUT_DRIVE.value
    ]
    search_out_media_cases = [
        case for case in current_cases
        if get_top_department(case.department.get_ancestors()).url == TopDepartments.S_OUT_MEDIA.value
    ]

    if (not taxi_kpb_cases
            and not zen_kpb_cases # На проде замьютить
            and not search_out_cases
            and not search_out_drive_cases
            and not search_out_media_cases
    ):
        drop_approvement()

    for case in taxi_kpb_cases:

        if case.is_equal(Cases.CHANGE_POSITION.value):
            approvers.extend(ApproverChains.get_hrbp(case))
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.CHANGE_ORG.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_SALARY.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_RATE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_OFFICE.value):
            pass

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_head(case))
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_head(case))
            approvers.extend(ApproverChains.get_hrbp(case))

        # TODO: Не забыть про развилку Даня -2 (когда доберемся до основной ветки)
        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_GIVE.value):
            if ApproverChains.first_common_dep_if_level_5(case):
                approvers.extend(ApproverChains.get_first_common_head(case))
                approvers.extend(ApproverChains.get_hrbp(case))
            else:
                approvers.extend(ApproverChains.get_hrbp(case))
            # approvers.extend(ApproverChains.get_daniilsh_minus2(case))

        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_ACCEPT.value):
            if ApproverChains.first_common_dep_if_level_5(case):
                approvers.extend(ApproverChains.get_first_common_head(case))
                approvers.extend(ApproverChains.get_hrbp(case))
            else:
                approvers.extend(ApproverChains.get_hrbp(case))
            # approvers.extend(ApproverChains.get_daniilsh_minus2(case))

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_GIVE.value):
            # approvers.extend(ApproverChains.get_hrbp(case))
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_ACCEPT.value):
            # approvers.extend(ApproverChains.get_hrbp(case))
            approvers.extend(ApproverChains.get_accepting_head(case))

    for case in zen_kpb_cases:
        if case.is_equal(Cases.CHANGE_POSITION.value):
            approvers.extend(ApproverChains.get_hrbp(case))
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.CHANGE_ORG.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_SALARY.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_RATE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.CHANGE_OFFICE.value):
            pass

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_head(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_GIVE.value):
            approvers.extend(ApproverChains.get_head(case))
            approvers.extend(ApproverChains.get_hrbp(case))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_ACCEPT.value):
            approvers.extend(ApproverChains.get_head(case))
            approvers.extend(ApproverChains.get_hrbp(case))

    for case in search_out_cases:
        ancestors = case.department.get_ancestors()

        responsible_ones = dict([
            ('as_dep08316', 'lmullayanova'),
            ('as_7861_dep81328', 'lmullayanova'),
            ('outstaff_3210_2341_9150_6369', 'leeuw'),
            ('outstaff_3210_2341_9150_dep82813', 'osuvorova'),
            ('as_dep98708', 'klimenko'),
            ('as_dep78081', 'klimenko'),
            ('as_dep28876', 'klimenko'),
            ('as_dep19389', 'klimenko'),
            ('as_dep13817', 'derealty'),
            ('as_dep86432', 'chihsv'),
            ('as_7861_dep07752', 'anna-ti'),
            ('ext_data', 'gaiazov')
        ])

        responsibles = {responsible_ones.get(dep.url) for dep in ancestors}
        responsibles.discard(None)

        if case.is_equal(Cases.CHANGE_POSITION.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.CHANGE_ORG.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.CHANGE_SALARY.value):
            if case.person_changes.is_salary_piecework:
                pass
            else:
                approvers.extend(list(responsibles))
                approvers.extend(['anna-ti'])

        if case.is_equal(Cases.CHANGE_RATE.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.CHANGE_OFFICE.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_ACCEPT.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.TRANSFER_FROM_MATERNITY_GIVE.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_ACCEPT.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_CHANGE_BRANCH_GIVE.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_ACCEPT.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_FROM_INTERN_GIVE.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_GIVE.value):
            pass

        if case.is_equal(Cases.TRANSFER_WITH_BUDGET_ACCEPT.value):
            pass

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_GIVE.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.TRANSFER_WO_BUDGET_ACCEPT.value):
            approvers.extend(list(responsibles))

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_GIVE.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WO_BUDGET_ACCEPT.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_GIVE.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

        if case.is_equal(Cases.TRANSFER_CHANGE_BG_WITH_BUDGET_ACCEPT.value):
            approvers.extend(list(responsibles))
            approvers.extend(['anna-ti'])

    for case in search_out_drive_cases:
        approvers.extend(['mg'])

    for case in search_out_media_cases:
        approvers.extend(['shurochka'])

    exclude_heads = {'bunina', 'tigran', 'volozh', 'abovsky', 'daniilsh',
                     'styskin', 'dpolischuk', 'aleshin', 'rommich', 'dkachmar',
                     'asavinovsky', 'liumin', 'krasil', 'abash', 'izalesskiy',
                     'andykor', 'jpopelysheva', 'vmarchuk', 'dmasyuk', 'ivanov',
                     'mborisov', 'grishakov', 'romanmoroz', 'veral', 'demyashkevich'}

    approvers = [head for head in approvers if head not in exclude_heads | {person_changes.person.login}]

    return approvers


# add_approver('mamasuevn')

if not DEV_MODE:
    approvers = main(params)
    for approver in approvers:
        add_approver(approver)

# remove_duplicates()
# set_approvement_field(is_parallel =False, text='Согласуй, пожалуйста, изменения. (этот ОК сгенерирован автоматически)')
