import os
import sqlite3
import logging
import datetime
from sqlite3 import Error
from file_import_utils import (
    isneed_updating_photo_file,
    send_email,
    prepare_for_logging,
    get_staff_logins,
)

LOG_FILE = f'{os.path.splitext(__file__)[0]}.log'
LOG_FILE_TO_SEND = 'db_result.txt'
MAIL_TO = ['diff_another@yandex-team.ru']
YAV_TOKEN = os.getenv('YAV_TOKEN', '')
PHOTO_DB = 'photo'


def create_connection(path):
    connection = None
    try:
        connection = sqlite3.connect(path)
        logging.info('Connection to SQLite DB successful')
    except Error as e:
        logging.error(f"The error '{e}' occurred")
    return connection


def execute_query(connection, query) -> None:
    cursor = connection.cursor()
    try:
        cursor.execute(query)
        connection.commit()
    except Error as e:
        logging.error(f"The error '{e}' occurred")
    return None


def execute_read_query(connection, query) -> None:
    cursor = connection.cursor()
    result = None
    try:
        cursor.execute(query)
        result = cursor.fetchall()
        return result
    except Error as e:
        logging.error(f"The error '{e}' occurred")
    return None


def get_id_pic(connection, login) -> int or None:
    select_item = 'SELECT id FROM pictures WHERE login = \'{}\' LIMIT 1;'.format(login)
    result = execute_read_query(connection, select_item)
    if result:
        return result[0][0]
    else:
        return None


def get_item_pic(connection, login) -> list or None:
    select_item = 'SELECT id, login, type, status, last_check, '\
                  'last_update FROM pictures WHERE login = \'{}\' LIMIT 1;'.format(login)
    result = execute_read_query(connection, select_item)
    if result:
        return result[0]
    else:
        return None


def update_pic(connection, db_item: list) -> None:
    if db_item[0]:
        sql = '''UPDATE pictures SET
            login = \'{}\', type = \'{}\', status  = \'{}\', last_check  = \'{}\',
             last_update = \'{}\' WHERE id = {}
            '''.format(db_item[1], db_item[2], db_item[3], db_item[4], db_item[5], db_item[0])
    else:
        sql = '''INSERT INTO pictures (login, type, status, last_check, last_update)
            VALUES (\'{}\', \'{}\', \'{}\', \'{}\', \'{}\');
            '''.format(db_item[1], db_item[2], db_item[3], db_item[4], db_item[5])
    execute_query(connection, sql)
    connection.commit()
    return None


def get_long_log(user: list) -> str:
    check = datetime.datetime.fromtimestamp(user[4]).strftime("%Y-%m-%d %H:%M:%S")
    update = datetime.datetime.fromtimestamp(user[5]).strftime("%Y-%m-%d %H:%M:%S")
    log = '{}({}), type={}, status={}, CHECK: {}, UPDATE: {}'
    return log.format(user[1], user[0], user[2], user[3], check, update)


def check_in(connection, login: str) -> bool:
    check = int(datetime.datetime.now().timestamp())
    is_need, binary_data = isneed_updating_photo_file(PHOTO_DB, login, update=True)
    pic = get_item_pic(connection, login)
    if not pic:
        pic = [None, login, 0, 0, check, check]
    else:
        pic = list(pic)
    if is_need and binary_data and pic[2] == 0:
        pic[3] = 0
        pic[4] = pic[5] = check
        update_pic(connection, pic)
        logging.info('Update: %s', get_long_log(pic))
        return True
    pic[4] = check
    update_pic(connection, pic)
    return False


def do_update(connection, login: str, bit_number: int) -> None:
    pic = get_item_pic(connection, login)
    if not pic:
        check = int(datetime.datetime.now().timestamp())
        pic = [None, login, 0, 0, check, check]
    else:
        pic = list(pic)
        pic[3] |= bit_number
    update_pic(connection, pic)
    return None


def pictures_for_update(connection, bit_number: int) -> dict:
    result_dict = {}
    select_items = "SELECT id, login, type, status, last_check, last_update FROM pictures "\
        "WHERE type = 0"
    items = execute_read_query(connection, select_items) or []
    for item in items:
        if item[3] & bit_number == 0:
            result_dict[item[1]] = item
    return result_dict


def out(connection) -> None:
    select_items = "SELECT id, login, type, status, last_check, last_update FROM pictures"
    items = execute_read_query(connection, select_items)
    for item in items:
        logging.info(get_long_log(item))
    return None


def init_db() -> sqlite3.Connection:
    connection = create_connection('pictures_db.sqlite3')
    create_table = """
    CREATE TABLE IF NOT EXISTS pictures (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        login TEXT NOT NULL,
        type INTEGER NOT NULL,
        status INTEGER NOT NULL,
        last_check INTEGER,
        last_update INTEGER,
        UNIQUE(login)
    );"""
    execute_query(connection, create_table)
    return connection


def main(connection) -> None:
    date_s = datetime.datetime.now()
    regions = [
        # {'department_group.id': 57661},
        {'official.staff_biometric_agreement': True},
        # {'official.affiliation': 'yandex'},
    ]
    staff_dict = get_staff_logins(regions)
    counter: int = 0
    for login in staff_dict:
        if check_in(connection, login):
            counter += 1
            logging.debug('%s: updated', login)
    logging.info('Всего staff: %i, обновлено %i', len(staff_dict), counter)
    logging.info('Затрачено всего: %s', datetime.datetime.now()-date_s)
    return None


if __name__ == '__main__':
    prepare_for_logging(LOG_FILE, LOG_FILE_TO_SEND)
    connection = init_db()
    main(connection)
    connection.commit()
    connection.close()
    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'
    send_email(MAIL_TO, 'Выгрузка для DB_UPDATE_PICs', body_message, [LOG_FILE_TO_SEND])
    logging.info('----------------------')
