import os
import re
import csv
import json
import pyodbc
import datetime
import requests
import get_api_staff
import smtplib
import logging
import collections
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from email.mime.application import MIMEApplication
from vault_client.instances import Production as VaultClient


def get_secret_from_yav(secret_id: str, token: str, oauth_token: str) -> str:
    return VaultClient(
        authorization='OAuth {}'.format(oauth_token),
        decode_files=True
    ).get_version(secret_id)['value'][token]


URL_PHOTO = 'https://center.yandex-team.ru'
YAV_TOKEN = os.getenv('YAV_TOKEN', '')
SERVER_SQL = {
    'lenrus': ['lenrus00.ld.yandex.ru,3433', 'robot-parkingbadge_sql',
               get_secret_from_yav('sec-01fnnq0y90c48ks91xs3kmse4b', 'robot-parkingbadge_sql', YAV_TOKEN)],
    'lenfin': ['lenfin.yandex-team.ru,1433', 'lenfin_reader_n',
               get_secret_from_yav('sec-01fgv6ye50z18544k5qwrc9p5v', 'lenfin_reader_n', YAV_TOKEN)],
}


def get_photo_bytes(dir_str: str, login: str, to_file_name: str = None, force_url=False) -> bytes or None:
    file_name = ''.join([dir_str, '/', to_file_name if to_file_name else login, '.jpg'])
    if os.path.isfile(file_name) and not force_url:
        with open(file_name, 'rb') as binary_file:
            return binary_file.read()
    url = f'{URL_PHOTO}/api/v1/user/{login}/photo/460.jpg'
    response = requests.get(url)
    if response.status_code == 200:
        header_png = b'\x89PNG'
        header_jpg = b'\xff\xd8\xff\xe0\x00\x10JFIF\x00'
        if header_png[:4] == response.content[:4] or header_jpg[:8] == response.content[:8]:
            if not force_url:
                update_photo(file_name, response.content)
            return response.content
        logging.error(f'Wrong format photo: {login}')
    return None


def update_photo(file_name: str, pic_data: bytes) -> None:
    with open(file_name, "wb") as binary_file:
        binary_file.write(pic_data)
    return None


def isneed_updating_photo_file(
        dir_str: str, login: str,
        to_file_name: str = None,
        update: bool = True
) -> (bool, bytes or None):
    file_name = ''.join([dir_str, '/', to_file_name if to_file_name else login, '.jpg'])
    pic_data = get_photo_bytes(dir_str, login, to_file_name, force_url=True)
    if pic_data:
        if os.path.isfile(file_name):
            file_size = os.path.getsize(file_name)
            file_size_update = len(pic_data)
            if file_size_update != file_size:
                if update:
                    update_photo(file_name, pic_data)
                return (True, pic_data)
            else:
                return (False, pic_data)
        else:
            if update:
                update_photo(file_name, pic_data)
            return (True, pic_data)
    return (False, None)


def load_from_lenel(file_name: str, code_page: str = 'utf-16-le') -> dict:
    try:
        csv_file = open(file_name, 'r', encoding=code_page)
        csv_reader = csv.reader(csv_file, delimiter=';', quotechar='"')
    except FileNotFoundError:
        logging.error('Файл не найден: %s', file_name)
        return {}
    data_read = [row for row in csv_reader]
    csv_file.close()
    result_dict: dict = {}
    for id_badge, last_name, first_name, login, type_bagde, date_activate, date_deactivate in data_read:
        result_dict[id_badge] = [
            login,
            first_name.replace("́", "").replace("̆", ""),
            last_name.replace("́", "").replace("̈", ""),
            datetime.datetime.strptime(date_activate, '%Y-%m-%d %H:%M:%S.%f'),
            datetime.datetime.strptime(date_deactivate, '%Y-%m-%d %H:%M:%S.%f'),
            type_bagde,
        ]
    return result_dict


def save_csv_like_lenel(result_dict: dict, file_name_csv: str) -> None:
    csv_file = open(file_name_csv, 'w', newline='', encoding='utf-8')
    csv_writer = csv.writer(csv_file, delimiter=';', quotechar='"', quoting=csv.QUOTE_MINIMAL)
    for id_badge, temp in result_dict.items():
        csv_writer.writerow([
            id_badge,
            temp[2],
            temp[1],
            temp[0],
            temp[5],
            temp[3].strftime('%Y-%m-%d %H:%M:%S.%f')[:-3],
            temp[4].strftime('%Y-%m-%d %H:%M:%S.%f')[:-3],
        ])
    csv_file.close()
    return None


def get_sql_request_from_lenels(sql_name: str, query_string: str) -> tuple:
    server_name = SERVER_SQL[sql_name]
    auth_string = 'Trusted_Connection=yes;' if not server_name[1] else f'UID={server_name[1]};PWD={server_name[2]}'
    cnxn = pyodbc.connect(f'Driver={{SQL Server}};Server={server_name[0]};Database=ACCESSCONTROL;{auth_string}')
    cursor = cnxn.cursor()
    cursor.execute(query_string)
    return cursor.fetchall()


def get_accesslevels_from_lenel(sql_name: str) -> dict:
    query_string = """SET NOCOUNT ON;
      SELECT [ACCESSCONTROL].[dbo].[ACCESSLVL].[ACCESSLVID]
      ,[ACCESSCONTROL].[dbo].[ACCESSLVL].[DESCRIPT]
      ,[ACCESSCONTROL].[dbo].[ACCLVLINK].[PANELID]
      ,[ACCESSCONTROL].[dbo].[ACCLVLINK].[READERID]
      ,[ACCESSCONTROL].[dbo].[READER].[READERDESC]
      FROM [ACCESSCONTROL].[dbo].[ACCESSLVL] WITH (nolock)
      INNER JOIN [ACCESSCONTROL].[dbo].[ACCLVLINK] ON
      [ACCESSCONTROL].[dbo].[ACCESSLVL].[ACCESSLVID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[ACCESSLVID]
      INNER JOIN [ACCESSCONTROL].[dbo].[READER] ON
      ([ACCESSCONTROL].[dbo].[READER].[PANELID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[PANELID] AND
      [ACCESSCONTROL].[dbo].[READER].[READERID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[READERID])"""
    result_dict: dict = {}
    items = get_sql_request_from_lenels(sql_name, query_string)
    for accesslvlid, descript, panelid, readerid, readerdesc in items:
        if readerdesc.find('(') < 0:
            continue
        room = readerdesc[readerdesc.find('(')+1:readerdesc.find(')')]
        if room is not None and len(room) > 1:
            if accesslvlid in result_dict:
                result_dict[accesslvlid]['rooms'].add(room)
            else:
                result_dict[accesslvlid] = {'rooms': {room}, 'name': descript}
    return result_dict


def get_badge_from_lenel(sql_name: str) -> dict:
    # {id_badge [login, first_name, last_name, date_activate, date_deactivate, type_bagde]}
    query_string = """SET NOCOUNT ON;
        SELECT [ACCESSCONTROL].[dbo].[BADGE].[ID]
              ,[ACCESSCONTROL].[dbo].[EMP].[LASTNAME]
              ,[ACCESSCONTROL].[dbo].[EMP].[FIRSTNAME]
              ,[ACCESSCONTROL].[dbo].[EMP].[MIDNAME]
              ,[ACCESSCONTROL].[dbo].[BADGE].[TYPE]
              ,[ACCESSCONTROL].[dbo].[BADGE].[ACTIVATE]
              ,[ACCESSCONTROL].[dbo].[BADGE].[DEACTIVATE]
        FROM [ACCESSCONTROL].[dbo].[BADGE] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[EMP] ON
              [ACCESSCONTROL].[dbo].[BADGE].[EMPID] = [ACCESSCONTROL].[dbo].[EMP].[ID]
        WHERE [STATUS] = 1 AND [DEACTIVATE] > CURRENT_TIMESTAMP AND [ACCESSCONTROL].[dbo].[BADGE].[ID]>34000"""
    result_dict: dict = {}
    items = get_sql_request_from_lenels(sql_name, query_string)
    for id_badge, last_name, first_name, login, type_bagde, date_activate, date_deactivate in items:
        result_dict[str(id_badge).replace('\ufeff', '')] = [
            login if isinstance(login, str) else 'NULL',
            first_name.replace("́", "").replace("̆", "") if isinstance(first_name, str) else 'NULL',
            last_name.replace("́", "").replace("̈", "") if isinstance(last_name, str) else 'NULL',
            date_activate,
            date_deactivate,
            type_bagde,
        ]
    return result_dict


def get_allbadge_from_lenel(sql_name: str) -> dict:
    # All badge with login in MIDNAME: {login: {id_badge1, id_badge2}}
    query_string = """SET NOCOUNT ON;
        SELECT [ACCESSCONTROL].[dbo].[BADGE].[ID]
              ,[ACCESSCONTROL].[dbo].[EMP].[LASTNAME]
              ,[ACCESSCONTROL].[dbo].[EMP].[FIRSTNAME]
              ,[ACCESSCONTROL].[dbo].[EMP].[MIDNAME]
              ,[ACCESSCONTROL].[dbo].[BADGE].[TYPE]
              ,[ACCESSCONTROL].[dbo].[BADGE].[ACTIVATE]
              ,[ACCESSCONTROL].[dbo].[BADGE].[DEACTIVATE]
        FROM [ACCESSCONTROL].[dbo].[BADGE] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[EMP] ON
              [ACCESSCONTROL].[dbo].[BADGE].[EMPID] = [ACCESSCONTROL].[dbo].[EMP].[ID]
        WHERE [STATUS] = 1 AND [DEACTIVATE] > CURRENT_TIMESTAMP"""
    result_dict = collections.defaultdict(set)
    rgx = re.compile(r'^[a-z0-9]+([a-z-]+[a-z0-9-]+|[a-z0-9-]+[a-z-]+)$')
    items = get_sql_request_from_lenels(sql_name, query_string)
    for id_badge, last_name, first_name, login, type_bagde, date_activate, date_deactivate in items:
        if isinstance(login, str) and rgx.match(login):
            result_dict[login].add(str(id_badge))
    return dict(result_dict)


def send_email(mail_to: list, subject: str, body: str, file_names: list) -> None:
    MAIL_HOST = 'outbound-relay.yandex.net'
    MAIL_FROM = 'noreply@yandex-team.ru'
    msg = MIMEMultipart()
    msg['Subject'] = subject
    msg['From'] = MAIL_FROM
    msg['To'] = ', '.join(mail_to)
    # msg['X-Priority'] = '1'
    # msg['X-MSMail-Priority'] = 'High'
    # msg['Importance'] = 'High'
    msg.attach(MIMEText(body, 'html', _charset='utf-8'))
    for file_name in file_names:
        with open(file_name, 'rb') as fh:
            attach = MIMEApplication(fh.read(), Name=file_name, _subtype='csv')
        attach['Content-Disposition'] = 'attachment; filename="%s"' % file_name
        msg.attach(attach)
    conn = smtplib.SMTP(MAIL_HOST)
    conn.sendmail(MAIL_FROM, mail_to, msg.as_string())
    conn.quit()
    return None


def get_staff_logins(regions: list) -> dict:
    result_dict = {}
    for region in regions:
        try:
            persons_filter = {
                'official.is_robot': False,
                '_fields': ','.join(['id', 'login', 'name',
                                     'location.office.id',
                                     'official.affiliation']),
            }
            persons_filter.update(region)
            for login, api_data in get_api_staff.staff_api_persons(persons_filter).items():
                result_dict[login] = [
                        {},
                        api_data['name']['first']['ru'],
                        ' ',  # api_data['name']['last']['ru'],
                        api_data['location']['office']['id'],
                        api_data['official']['affiliation'],
                ]
        except requests.exceptions.SSLError as error:
            logging.error('Ошибка соединения со СТАФФ: %s', error)
            raise
    return result_dict


def read_white_list(file_name: str) -> dict:
    csv_file = open(file_name, 'r', newline='', encoding='cp1251')
    csv_reader = csv.reader(csv_file, delimiter=';', quotechar='"')
    data_read = [row for row in csv_reader]
    csv_file.close()
    result_dict: dict = {}
    for _, _, data_read, _, _, _, _, _, first_name, last_name, hi_card, lo_card, _, \
            COMPANY_NAME, status, id_number, _, total_number, _, _, _, _, plus in data_read:
        hi_card = hi_card.replace("\xa0", "")
        lo_card = lo_card.replace("\xa0", "")
        id_badge = 65536 * int(hi_card) + int(lo_card)
        if status == 'USED' and plus == '+' and id_badge != 0:
            result_dict[str(id_badge)] = [first_name, last_name, status, hi_card, lo_card, plus]
    return result_dict


def save_data_to_file(my_details: dict, filename: str) -> None:
    filename = f'{filename}.dat'
    with open(filename, 'w', encoding='utf-8') as json_file:
        json.dump(my_details, json_file, ensure_ascii=False)
    return None


def load_data_from_file(filename: str) -> dict:
    filename = f'{filename}.dat'
    try:
        with open(filename, 'r', encoding='utf-8') as json_file:
            json_data = json.load(json_file)
    except FileNotFoundError:
        logging.error('Файл не найден: %s', filename)
        return {}
    return json_data


def prepare_for_logging(log_file: str, log_file_to_send: str) -> None:
    format_str = '%(asctime)s [%(levelname)s]: %(message)s'
    logging.basicConfig(filename=log_file, format=format_str, level=logging.INFO)
    handler = logging.FileHandler(filename=log_file_to_send, mode='w')
    handler.setFormatter(logging.Formatter(format_str))
    logging.getLogger('root').addHandler(handler)
    return None


def get_log_statictic(previous_set: set, present_set: set, whitelist_set: set = None) -> None:
    whitelist_set = whitelist_set or set()
    logging.info(f'Было: {len(previous_set)} У нас: {len(present_set)} Белый список: {len(whitelist_set)}')
    logging.info(f'Объединение: {len((previous_set & present_set)-whitelist_set)} '
                 f'Удаляем: {len((previous_set-present_set)-whitelist_set)} '
                 f'Добавляем: {len((present_set-previous_set)-whitelist_set)}')
    return None


if __name__ == '__main__':
    pass
