import os
import csv
import requests
import logging

OAUTH_TOKEN = os.getenv('ROBOT_OAUTH_TOKEN', '')

STAFFAPI_HOST = {
    'production': 'staff-api.yandex-team.ru',
    'testing': 'staff-api.test.yandex-team.ru',
}[os.getenv('ENV_TYPE', 'testing')]

STAFF_HOST = {
    'production': 'staff.yandex-team.ru',
    'testing': 'staff.test.yandex-team.ru',
}[os.getenv('ENV_TYPE', 'testing')]


# 0 - Бейдж сотрудника и у него есть guid
# 1 - Бейдж кандидата или резерв
# 2 - Бейдж анонима и стоит признак anonym_food_allowed
# BADGE_TYPE = {'0': 'employee', '1': 'candidate', '2': 'anonym'}


def sanitize(token: str) -> str:
    return token.strip()


def staff_api_departments(filter_params: dict = {}) -> dict:
    """{login: {field: value}}"""
    staffapi_session = requests.Session()
    staffapi_session.headers.update({'Authorization': f'OAuth {OAUTH_TOKEN}'})
    persons_filter = {
        'person.official.is_dismissed': False,
        '_fields': ','.join(['person.id', 'person.login', 'person.name']),
        '_nopage': 1,
        '_sort': 'person.id',
        '_limit': 500,
    }
    persons_filter.update(filter_params)
    last_id = 0
    count = 0
    result_data = {}
    while True:
        persons_filter['_query'] = f'person.id>{last_id}'
        response = staffapi_session.get(
            url=f'https://{STAFFAPI_HOST}/v3/departmentstaff',
            params=persons_filter,
        )
        if response.status_code != 200:
            raise RuntimeError(f'{response.status_code}: {response.content}')

        items = response.json()['result']
        if items:
            for person_dict in items:
                result_data[person_dict['person']['login']] = person_dict
            last_id = items[-1]['person']['id']
        else:
            break
        count += 1
    logging.debug(f'{count} запросов в staff-api сделано')
    return result_data


def staff_api_persons(filter_params: dict = {}) -> dict:
    """{login: {field: value}}"""
    staffapi_session = requests.Session()
    staffapi_session.headers.update({'Authorization': f'OAuth {OAUTH_TOKEN}'})
    persons_filter = {
        'official.is_dismissed': False,
        '_fields': ','.join(['id', 'login', 'name', 'location.office.code', 'cars.plate', 'cars.model']),
        '_nopage': 1,
        '_sort': 'id',
        '_limit': 500,
    }
    persons_filter.update(filter_params)
    last_id = 0
    count = 0
    result_data = {}
    while True:
        persons_filter['_query'] = f'id>{last_id}'
        response = staffapi_session.get(
            url=f'https://{STAFFAPI_HOST}/v3/persons',
            params=persons_filter,
        )
        if response.status_code != 200:
            raise RuntimeError(f'{response.status_code}: {response.content}')
        items = response.json()['result']
        if items:
            for person_dict in items:
                result_data[person_dict['login']] = person_dict
            last_id = items[-1]['id']
        else:
            break
        count += 1
    logging.debug(f'{count} запросов в staff-api сделано')
    return result_data


def rfid_badges() -> dict:
    """{login: badge_code}"""
    rfid_session = requests.Session()
    rfid_session.headers.update({'Authorization': f'OAuth {OAUTH_TOKEN}'})

    response = rfid_session.get(url=f'https://{STAFF_HOST}/rfid-api/export/')
    if response.status_code != 200:
        raise RuntimeError(f'{response.status_code}: {response.content}')

    csv_reader = csv.reader(response._content.decode().splitlines(), delimiter=';')

    rfid_data = {}
    for login, guid, _, badge_code, is_active_str, first_name, last_name, owner, badge_type in csv_reader:
        if login and sanitize(is_active_str) == '1' and sanitize(badge_type) == '0':
            rfid_data[sanitize(login)] = sanitize(badge_code)
    return rfid_data


def get_data(filter_params: dict = {}) -> list:
    api_persons = staff_api_persons(filter_params)
    badges = rfid_badges()

    result = []
    for login, api_data in api_persons.items():
        result.append(
            (
                login,
                api_data['name']['first']['ru'],
                api_data['name']['last']['ru'],
                api_data['location']['office']['id'],
                badges.get(login, ''),
                api_data['cars'],
            )
        )
    return result
