import os
import re
import logging
import collections
from file_import_utils import (
    get_secret_from_yav,
    send_email,
    prepare_for_logging,
    get_sql_request_from_lenels,
)

LOG_FILE = f'{os.path.splitext(__file__)[0]}.log'
LOG_FILE_TO_SEND = 'lenel_readers_result.txt'
MAIL_TO = ['diff_another@yandex-team.ru']
YAV_TOKEN = os.getenv('YAV_TOKEN', '')
OAUTH_TOKEN = get_secret_from_yav('sec-01f35s2537s2d6sed431k7a8kj', 'oauth-token', YAV_TOKEN)


def get_error_time_panel_from_lenel(sql_name: str) -> list:
    query_string = """SET NOCOUNT ON;
        SELECT TOP (1000)[ACCESSCONTROL].[dbo].[LASTLOCATION].[EVENT_TIME_UTC]
        ,[ACCESSCONTROL].[dbo].[READER].[PANELID]
        ,[ACCESSCONTROL].[dbo].[READER].[READERID]
        ,[ACCESSCONTROL].[dbo].[READER].[READERDESC]
        ,[ACCESSCONTROL].[dbo].[LASTLOCATION].[EVENTID]
        ,[ACCESSCONTROL].[dbo].[LASTLOCATION].[EVENTTYPE]
        ,[ACCESSCONTROL].[dbo].[LASTLOCATION].[EMPID]
        ,[ACCESSCONTROL].[dbo].[LASTLOCATION].[ACCESS_FLAG]
        FROM [ACCESSCONTROL].[dbo].[LASTLOCATION] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[READER] ON
        ([ACCESSCONTROL].[dbo].[LASTLOCATION].[PANELID] = [ACCESSCONTROL].[dbo].[READER].[PANELID] AND
        [ACCESSCONTROL].[dbo].[LASTLOCATION].[READERID] = [ACCESSCONTROL].[dbo].[READER].[READERID])
        WHERE EVENT_TIME_UTC > CURRENT_TIMESTAMP OR EVENT_TIME_UTC < '2020-01-01 00:00:00.000'"""
    items = get_sql_request_from_lenels(sql_name, query_string)
    result_list = []
    for event_time, panelid, readerid, readerdesc, event_id, event_type, emp_id, access_flag in items:
        result_list.append((event_time, panelid, readerid, readerdesc, event_id, event_type, emp_id, access_flag))
    return result_list


def get_error_dublebadge_from_lenel(sql_name: str) -> dict:
    query_string = """SET NOCOUNT ON;
        SELECT [ACCESSCONTROL].[dbo].[BADGE].[ID]
            ,[ACCESSCONTROL].[dbo].[EMP].[LASTNAME]
            ,[ACCESSCONTROL].[dbo].[EMP].[FIRSTNAME]
            ,[ACCESSCONTROL].[dbo].[EMP].[MIDNAME]
            ,[ACCESSCONTROL].[dbo].[BADGE].[TYPE]
            ,[ACCESSCONTROL].[dbo].[BADGE].[ACTIVATE]
            ,[ACCESSCONTROL].[dbo].[BADGE].[DEACTIVATE]
        FROM [ACCESSCONTROL].[dbo].[BADGE] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[EMP] ON
            [ACCESSCONTROL].[dbo].[BADGE].[EMPID] = [ACCESSCONTROL].[dbo].[EMP].[ID]
        WHERE [STATUS] = 1 AND [DEACTIVATE] > CURRENT_TIMESTAMP AND [ACCESSCONTROL].[dbo].[EMP].[ID] IN
        (
        SELECT [ACCESSCONTROL].[dbo].[EMP].[ID]
        FROM [ACCESSCONTROL].[dbo].[BADGE] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[EMP] ON
            [ACCESSCONTROL].[dbo].[BADGE].[EMPID] = [ACCESSCONTROL].[dbo].[EMP].[ID]
        WHERE [STATUS] = 1 AND [DEACTIVATE] > CURRENT_TIMESTAMP
        AND [ACCESSCONTROL].[dbo].[BADGE].[ID]>34000
        GROUP BY [ACCESSCONTROL].[dbo].[EMP].[ID]
        HAVING count(*) >1)"""
    items = get_sql_request_from_lenels(sql_name, query_string)
    result_dict = collections.defaultdict(set)
    for id_badge, last_name, first_name, login, type_bagde, date_activate, date_deactivate in items:
        result_dict[id_badge] = [
            login if isinstance(login, str) else 'NULL',
            first_name if isinstance(first_name, str) else 'NULL',
            last_name if isinstance(last_name, str) else 'NULL',
            # date_activate,
            # date_deactivate,
            # type_bagde,
        ]
    return result_dict


def get_accesslevels_from_lenel(sql_name: str) -> dict:
    query_string = """SET NOCOUNT ON;
        SELECT [ACCESSCONTROL].[dbo].[ACCLVLINK].[PANELID]
        ,[ACCESSCONTROL].[dbo].[ACCLVLINK].[READERID]
        ,[ACCESSCONTROL].[dbo].[READER].[READERDESC]
        ,[ACCESSCONTROL].[dbo].[ACCESSLVL].[ACCESSLVID]
        ,[ACCESSCONTROL].[dbo].[ACCESSLVL].[DESCRIPT]
        FROM [ACCESSCONTROL].[dbo].[READER] WITH (nolock)
        INNER JOIN [ACCESSCONTROL].[dbo].[ACCLVLINK] ON
        ([ACCESSCONTROL].[dbo].[READER].[PANELID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[PANELID] AND
        [ACCESSCONTROL].[dbo].[READER].[READERID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[READERID])
        INNER JOIN [ACCESSCONTROL].[dbo].[ACCESSLVL] ON
        [ACCESSCONTROL].[dbo].[ACCESSLVL].[ACCESSLVID] = [ACCESSCONTROL].[dbo].[ACCLVLINK].[ACCESSLVID]
        ORDER BY [ACCESSCONTROL].[dbo].[ACCLVLINK].[PANELID]
        ,[ACCESSCONTROL].[dbo].[ACCLVLINK].[READERID]"""
    items = get_sql_request_from_lenels(sql_name, query_string)
    result_dict = collections.defaultdict(set)
    rgx = re.compile(r'коммутационная|серверная|КЦ|ЭКЦ|ГКЦ|серв')
    for panelid, readerid, readerdesc, accesslvlid, descript in items:
        panelid = str(panelid)
        readerid = str(readerid)
        # readerdesc = readerdesc.encode().decode('UTF8')
        if rgx.search(readerdesc) and accesslvlid not in (2, 9999):
            panelid = str(panelid)
            readerid = str(readerid)
            result_dict['-'.join([panelid, readerid])].add((accesslvlid, descript, readerdesc))
    return dict(result_dict)


def logging_out_levels(all_access_levels: dict) -> None:
    if all_access_levels:
        for data in all_access_levels.values():
            compact_level_data = []
            show_level = False
            for access_level in data:
                if access_level[0] == 23:
                    show_level = True
                    continue
                name_str = access_level[2]
                compact_level_data.append(f'{access_level[0]}:{access_level[1]}')
            if show_level and compact_level_data:
                logging.info('%s (%s)', name_str, ', '.join(compact_level_data))
    return None


def logging_out_badges(all_badges: dict) -> None:
    if all_badges:
        logging.info('----------------------')
        for key, data in all_badges.items():
            logging.info('%s (%s)', key, ', '.join(data))
    return None


def logging_out_times(all_times: list) -> None:
    if all_times:
        logging.info('----------------------')
        for data in all_times:
            logging.info(', '.join(str(x) for x in data))
    return None


def main() -> None:
    all_access_levels = get_accesslevels_from_lenel('lenrus')
    logging_out_levels(all_access_levels)
    logging.info(f'Всего урвней: {len(all_access_levels)}')
    all_error_badge = get_error_dublebadge_from_lenel('lenrus')
    logging_out_badges(all_error_badge)
    logging.info(f'Всего бейджей: {len(all_error_badge)//2}')
    all_error_time = get_error_time_panel_from_lenel('lenrus')
    logging_out_times(all_error_time)
    logging.info(f'Всего ошибок времени: {len(all_error_time)}')
    return None


if __name__ == '__main__':
    prepare_for_logging(LOG_FILE, LOG_FILE_TO_SEND)
    main()
    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'
    send_email(MAIL_TO, 'Выгрузка для READERS_LEVELS и ERROR_BADGES', body_message, [LOG_FILE_TO_SEND])
    send_email(['startrek@yandex-team.ru'], 'ACVS-8852', body_message, [LOG_FILE_TO_SEND])
    logging.info('----------------------')
