import os
import re
import csv
import logging
import datetime
import argparse
from get_api_staff import staff_api_persons
from file_import_utils import (
    send_email,
    prepare_for_logging,
    load_from_lenel,
    save_csv_like_lenel,
    read_white_list,
    get_badge_from_lenel,
    get_log_statictic,
)

LOG_FILE = f'{os.path.splitext(__file__)[0]}.log'
LOG_FILE_TO_SEND = 'avrora_result.txt'
MAIL_TO = ['diff_another@yandex-team.ru']
DIRS = 'avrora'


def sanitize(string_in: str) -> str:
    return string_in.encode('cp1251', errors='ignore').decode('cp1251')


def read_accesscontrolcardshort(file_name: str, date_now: datetime) -> dict:
    csv_file = open(file_name, 'r', newline='', encoding='cp1251')
    csv_reader = csv.reader(csv_file, delimiter=',', quotechar='"')
    data_read = [row for row in csv_reader]
    csv_file.close()
    result_dict: dict = {}
    for _, _, data_read, _, _, _, _, _, first_name, last_name, hi_card, lo_card, _, \
            COMPANY_NAME, status, id_number, _, total_number, _, _, _, _ in data_read:
        hi_card = hi_card.replace("\xa0", "")
        lo_card = lo_card.replace("\xa0", "")
        id_badge = 65536 * int(hi_card) + int(lo_card)
        if status == 'USED' and id_badge != 0:
            result_dict[str(id_badge)] = [
                'no_login',
                first_name,
                last_name,
                date_now,
                date_now,
            ]
    return result_dict


def load_from_ernstyoung(file_name: str) -> dict:
    result_dict = {}
    csv_file = open(file_name, 'r', encoding='cp1251')
    csv_reader = csv.reader(csv_file, delimiter=';', quotechar='"')
    data_read = [row for row in csv_reader]
    csv_file.close()
    for _, login, id_badge in data_read:
        if len(id_badge) < 6:
            logging.error(f' {login}: {id_badge[0:-5]} {id_badge[-5:]}')
            continue
        ya_badge = 65536 * int(id_badge[0:-5]) + int(id_badge[-5:])
        result_dict[str(ya_badge)] = [login]
    return result_dict


def compare_data_abcf(staff_dict: dict, file_name: str, date_now: datetime) -> None:
    staff_set = set(staff_dict)
    HEADER_CSV = ['COMPANY', 'NAME', 'LAST NAME',
                  'UNIQUE ID(LOGIN)',
                  'DATE START', 'DATE END',
                  'JOB CODE', 'CARD NUMBER',
                  'ACTION', 'AREA', 'AREA']
    COMPANY_NAME = 'Яндекс'
    AREA_NUMBER_1 = 'ЯНДЕКС'
    AREA_NUMBER_2 = 'ДВЕРИ ПО ПЕРИМЕТРУ 1-ого ЭТАЖА'
    WHITE_LIST_DATA_FILE_NAME = f'{DIRS}/AvroraABCF_not_delete20210913.csv'
    PREVIOUS_DATA_FILE_NAME = f'{DIRS}/cardImport_ABCF_previous.dat'
    AVRORA_ABCF_DATA_FILE_NAME = f'{DIRS}/accesscontrolcardshort20210913.csv'
    MAIL_TO_SEND = {
        'production': ['diff_avrora-abcf@yandex-team.ru'],
        'testing': ['helther@yandex-team.ru'],
    }[os.getenv('ENV_TYPE', 'testing')]

    badgewhitelist_dict = read_white_list(WHITE_LIST_DATA_FILE_NAME)
    badgewhitelist_set = set(badgewhitelist_dict)

    previous_dict = load_from_lenel(PREVIOUS_DATA_FILE_NAME, code_page='utf-8')
    if not previous_dict:
        previous_dict = read_accesscontrolcardshort(AVRORA_ABCF_DATA_FILE_NAME, date_now)
    previous_set = set(previous_dict)

    region = {'location.office.code': 'avrora'}  # {'department_group.id': 57661}
    add_set = set(staff_api_persons(region))

    file_name_csv = f"{file_name}{date_now.strftime('%Y%m%d%H%M%S')}.csv"
    csv_file = open(file_name_csv, 'w', newline='', encoding='cp1251')
    spamwriter = csv.writer(csv_file, delimiter=';', quotechar='"', quoting=csv.QUOTE_MINIMAL)
    spamwriter.writerow(HEADER_CSV)

    to_do = 'delete'
    for id_badge in (previous_set-staff_set)-badgewhitelist_set:
        try:
            id_badge_i = int(id_badge)
            spamwriter.writerow([
                COMPANY_NAME,
                sanitize(previous_dict[id_badge][1]),
                sanitize(previous_dict[id_badge][2]),
                previous_dict[id_badge][0],
                previous_dict[id_badge][3].strftime('%Y/%m/%d %H:%M:%S'),
                previous_dict[id_badge][4].strftime('%Y/%m/%d %H:%M:%S'),
                str(id_badge_i >> 16),
                str(id_badge_i & 0xFFFF).zfill(5),
                to_do,
                AREA_NUMBER_1,
                AREA_NUMBER_2,
            ])
        except ValueError:
            logging.error(f'Bad badge: {id_badge}: {previous_dict[id_badge]}')
            continue
        except UnicodeEncodeError:
            logging.error(f'Bad symbol: {id_badge}: {previous_dict[id_badge]}')
            raise

    to_do = 'add'
    for id_badge in (staff_set-previous_set)-badgewhitelist_set:
        if staff_dict[id_badge][0] in add_set:
            try:
                id_badge_i = int(id_badge)
            except ValueError:
                logging.error(f'Bad badge: {id_badge}: {staff_dict[id_badge]}')
                continue
            spamwriter.writerow([
                COMPANY_NAME,
                sanitize(staff_dict[id_badge][1]),
                sanitize(staff_dict[id_badge][2]),
                staff_dict[id_badge][0],
                staff_dict[id_badge][3].strftime('%Y/%m/%d %H:%M:%S'),
                staff_dict[id_badge][4].strftime('%Y/%m/%d %H:%M:%S'),
                str(id_badge_i >> 16),
                str(id_badge_i & 0xFFFF).zfill(5),
                to_do,
                AREA_NUMBER_1,
                AREA_NUMBER_2,
            ])
    csv_file.close()
    get_log_statictic(previous_set, staff_set, badgewhitelist_set)
    body_message = 'Файлы приложены как вложения к данному письму. '\
        f'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>{file_name_csv}'
    if os.getenv('ENV_TYPE', 'testing') == 'production' and args.send:
        save_csv_like_lenel(staff_dict, PREVIOUS_DATA_FILE_NAME)
        send_email(MAIL_TO_SEND, 'Выгрузка пропусков для БЦ Аврора ABCF',
                   body_message, [file_name_csv])
    return None


def compare_data_d_ey(staff_dict: dict, file_name_str: str, date_now: datetime) -> None:
    staff_set = set(staff_dict)
    COMPANY_NAME = 'Yandex'
    TAIL_ROW = ['0', '', '1', 'Sublease_Yandex', '', '', '', 'Yandex']
    PREVIOUS_DATA_FILE_NAME = f'{DIRS}/cardImport_EY_previous.dat'
    # WHITE_LIST_DATA_FILE_NAME = f'{DIRS}/AvroraEY_not_delete20210913.csv'
    AVRORA_EY_DATA_FILE_NAME = f'{DIRS}/Card Holder 9-8-2021 6-19-35 PM.csv'
    MAIL_TO_SEND = {
        'production': ['diff_avrora-ey@yandex-team.ru'],
        'testing': ['helther@yandex-team.ru'],
    }[os.getenv('ENV_TYPE', 'testing')]

    badgewhitelist_set = set()
    previous_dict = load_from_lenel(PREVIOUS_DATA_FILE_NAME, code_page='utf-8')
    if not previous_dict:
        previous_dict = load_from_ernstyoung(AVRORA_EY_DATA_FILE_NAME)
    previous_set = set(previous_dict)

    file_name_del = f"{file_name_str}{date_now.strftime('%Y%m%d%H%M%S')}_del.csv"
    csv_file = open(file_name_del, 'w', newline='', encoding='cp1251')
    spamwriter = csv.writer(csv_file, delimiter=',', quotechar='"', quoting=csv.QUOTE_MINIMAL)
    for id_badge in (previous_set-staff_set):
        try:
            id_badge_i = int(id_badge)
        except ValueError:
            logging.error(f'Bad badge: {id_badge}: {previous_dict[id_badge]}')
            continue
        card_id_str = ''.join([str(id_badge_i >> 16), str(id_badge_i & 0xFFFF).zfill(5)])
        spamwriter.writerow([COMPANY_NAME, previous_dict[id_badge][0], card_id_str])
    csv_file.close()

    duplicate_badge = []
    file_name_add = f"{file_name_str}{date_now.strftime('%Y%m%d%H%M%S')}_add.csv"
    csv_file = open(file_name_add, 'w', newline='', encoding='cp1251')
    spamwriter = csv.writer(csv_file, delimiter=',', quotechar='"', quoting=csv.QUOTE_MINIMAL)
    for id_badge in (staff_set-previous_set):
        try:
            id_badge_i = int(id_badge)
        except ValueError:
            logging.error(f'Bad badge: {id_badge}: {staff_dict[id_badge]}')
            continue
        card_id_str = ''.join([str(id_badge_i >> 16), str(id_badge_i & 0xFFFF).zfill(5)])
        row_list = [COMPANY_NAME, staff_dict[id_badge][0], card_id_str]
        row_list.extend(TAIL_ROW)
        if re.search('^282[2-58].', card_id_str) or \
                re.search('^283[067].', card_id_str):
            out_string = ','.join([staff_dict[id_badge][0], card_id_str])
            logging.error(f'Возможный дубликат пропуска EY: {out_string} [{id_badge}]')
            duplicate_badge.append(','.join(row_list))
        else:
            spamwriter.writerow(row_list)
    csv_file.close()

    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'\
        'Возможные дубликаты пропусков: LOGIN AvroraID (YandexID):<br><br>'\
        f"{'<br>'.join(duplicate_badge)}"
    if os.getenv('ENV_TYPE', 'testing') == 'production' and args.send:
        save_csv_like_lenel(staff_dict, PREVIOUS_DATA_FILE_NAME)
        send_email(
            MAIL_TO_SEND,
            'Выгрузка пропусков для БЦ Аврора D (E&Y)',
            body_message,
            [file_name_add, file_name_del])
    get_log_statictic(previous_set, staff_set, badgewhitelist_set)
    return None


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Process....')
    parser.add_argument('-avrora_abcf', action='store_true')
    parser.add_argument('-avrora_d', action='store_true')
    parser.add_argument('-send', action='store_true', help='Real send e-mails!!!')
    args = parser.parse_args()

    prepare_for_logging(LOG_FILE, LOG_FILE_TO_SEND)
    CURRENT_DATA_FILE_NAME = f'{DIRS}/LENELAllactive'
    date_s = datetime.datetime.now()
    logging.info(f'Avrora ABCF + D(E&Y): {date_s.strftime("%Y-%m-%d %H:%M:%S")}')
    file_name_current = f"{CURRENT_DATA_FILE_NAME}{date_s.strftime('%Y%m%d')}.csv"
    staff_dict = load_from_lenel(file_name_current, code_page='utf-8')
    if not staff_dict:
        staff_dict = get_badge_from_lenel('lenrus')
        save_csv_like_lenel(staff_dict, file_name_current)
    if args.avrora_d:
        logging.info('Avrora D E&Y')
        compare_data_d_ey(staff_dict, f'{DIRS}/cardImport_EY_', date_s)
        logging.info(f'Затрачено: {datetime.datetime.now()-date_s}')
    if args.avrora_abcf:
        logging.info('Avrora ABCF')
        compare_data_abcf(staff_dict, f'{DIRS}/cardImport_ABCF_', date_s)
    logging.info(f'Затрачено всего: {datetime.datetime.now()-date_s}')
    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'
    send_email(MAIL_TO, 'Выгрузка для Avrora ABCF & E&Y', body_message, [LOG_FILE_TO_SEND])
    logging.info('----------------------')
