import os
import time
import json
import requests
import datetime
import logging
import hmac
import hashlib
import base64
from db_utils import init_db, pictures_for_update, do_update
from file_import_utils import (
    get_photo_bytes,
    get_secret_from_yav,
    send_email,
    prepare_for_logging,
    get_allbadge_from_lenel,
    get_staff_logins,
    get_log_statictic,
)

LOG_FILE = f'{os.path.splitext(__file__)[0]}.log'
LOG_FILE_TO_SEND = 'hik_result.txt'
MAIL_TO = ['diff_another@yandex-team.ru']
YAV_TOKEN = os.getenv('YAV_TOKEN', '')
SOURCE_URL = get_secret_from_yav('sec-01fpws9qva8pth0stf92hvaxj0', 'SOURCE_URL', YAV_TOKEN)
APP_KEY = get_secret_from_yav('sec-01fpws9qva8pth0stf92hvaxj0', 'APP_KEY', YAV_TOKEN)
APP_SECRET = get_secret_from_yav('sec-01fpws9qva8pth0stf92hvaxj0', 'APP_SECRET', YAV_TOKEN)
HIKCENTRAL = 1
# STAFF_GENDER = {'unknown': 0, 'male': 1, 'female': 2}
GROUP_ID = {
    'vip': '3', 'yandex': '4', 'external': '5',
    'no_badge': '6', 181: '7', 306: '8', 268: '9',
}
# {'avrora': 181, 'mskoko': 306, 'lotteplaza': 268}
PRIVILEGE_GROUP_ID = '4'  # 'Все входы'
PHOTO_DB = 'photo'


def avrora_badge(id_badge: str) -> str:
    id_badge_i = int(id_badge)
    return ''.join([str(id_badge_i >> 16), str(id_badge_i & 0xFFFF).zfill(5)])


def make_hik_request(url_part: str, body: dict, request_params: dict = None) -> list:
    request_params = request_params or {}
    try_count: int = 0
    while True:
        try_count += 1
        body_key = 'POST\n'\
                   '*/*\n'\
                   'application/json;charset=UTF-8\n'\
                   f'x-ca-key:{APP_KEY}\n'\
                   f'{url_part}'
        base64_message = base64.b64encode(hmac.new(key=APP_SECRET.encode('utf-8'),
                                                   msg=body_key.encode('utf-8'),
                                                   digestmod=hashlib.sha256).digest())
        payload = {
            'Accept': '*/*',
            'Content-Type': 'application/json;charset=UTF-8',
            'X-Ca-Key': APP_KEY,
            'X-Ca-Signature': base64_message,
            'X-Ca-Signature-Headers': 'x-ca-key',
        }
        payload.update(request_params)
        session = requests.Session()
        session.headers.update(payload)
        response = requests.post(
            url=f'https://{SOURCE_URL}{url_part}',
            headers=payload,
            data=json.dumps(body),
        )
        if response.status_code != 200:
            if response.status_code == 503 and try_count < 25:
                response_string = response.content.decode('UTF-8')
                logging.error(f'{try_count}: {response.status_code}: {response_string}')
                time.sleep(5*try_count)
            else:
                response_string = response.content.decode('UTF-8')
                raise RuntimeError(f'{response.status_code}: {response_string}')
        else:
            break
    if response.json()['code'] != '0':
        if response.json()['code'] == '131':
            logging.error(f"Exist card: {body['personName']}: {body['cards']}")
            return None
        logging.error(f'Hik: {body}: response: {response.json()}')
        raise RuntimeError(f'{response.json()}')
    return response.json()['data']


def hik_api_get_accesslist() -> dict:
    url_part = '/artemis/api/acs/v1/privilege/group'
    data = {
        'pageNo': 1,
        'pageSize': 100,
        'type': 1
    }
    return make_hik_request(url_part, data)['list']


def hik_api_add_accesslist(person_ids: list, access_grp: str) -> dict:
    url_part = '/artemis/api/acs/v1/privilege/group/single/addPersons'
    list_to_chage = []
    for person_id in person_ids:
        list_to_chage.append({'id': person_id})
    data = {
        'privilegeGroupId': access_grp,
        'type': 1,
        'list': list_to_chage
    }
    return make_hik_request(url_part, data)


def get_photo(dirs: str, file_name: str) -> str or None:
    binary_data = get_photo_bytes(dirs, file_name)
    if binary_data:
        return base64.b64encode(binary_data).decode('utf-8')
    logging.error(f'No photo: {file_name}')
    return None


def get_group_code(location: str, affiliation: str) -> str:
    if location in (306, 268):
        group_id = GROUP_ID.get(location, '6')
    else:
        group_id = GROUP_ID.get(affiliation, '6')
    return group_id


def hik_api_upd_person(login, hik_person: dict, staff: list) -> str or None:
    updating_fields = {}
    hik_card = set()
    if 'cards' not in hik_person:
        logging.error(f'DEL:no card: {hik_person}')
        hik_api_del_person(hik_person['personId'])
        return None
    for key in hik_person['cards']:
        hik_card.add(key['cardNo'])
    cards = []
    if set(staff[0]) != hik_card:
        for card_key in staff[0]:
            cards.append({'cardNo': card_key})
        updating_fields.update({'cards': cards})
    group_code = get_group_code(staff[3], staff[4])
    if group_code != hik_person['orgIndexCode']:
        updating_fields.update({'orgIndexCode': group_code})
    name_differs = hik_person['personFamilyName'] != staff[2] or hik_person['personGivenName'] != staff[1]
    if name_differs:
        date_s = datetime.datetime.now()
        date_e = date_s + datetime.timedelta(days=3600)
        updating_fields.update({
            'personName': ' '.join(staff[1:3]),
            'personFamilyName': staff[2],
            'personGivenName': staff[1],
            'email': f'{login}@yandex-team.ru',
            # 'remark': f'update_export {date_s.strftime("%Y-%m-%d %H:%M:%S")}',
            'beginTime': date_s.astimezone().replace(microsecond=0).isoformat(),
            'endTime': date_e.astimezone().replace(microsecond=0).isoformat()
        })
    if updating_fields:
        url_part = '/artemis/api/resource/v1/person/single/update'
        logging.info(f'Update: {login}({staff[0]}): HIK:{hik_person} STAFF:{staff}')
        hik_person.update(updating_fields)
        return make_hik_request(url_part, hik_person)
    return None


def hik_api_add_person(login: str, staff: list) -> str:
    url_part = '/artemis/api/resource/v1/person/single/add'
    date_s = datetime.datetime.now()
    date_e = date_s + datetime.timedelta(days=3600)
    cards = []
    for card_key in staff[0]:
        cards.extend([{'cardNo': card_key}])
    pic_data = get_photo(PHOTO_DB, login)
    data = {'personName': ' '.join(staff[1:3]),
            'personFamilyName': staff[2],
            'personGivenName': staff[1],
            'orgIndexCode': get_group_code(staff[3], staff[4]),
            'email': f'{login}@yandex-team.ru',
            # 'remark': f'update_export {date_s.strftime("%Y-%m-%d %H:%M:%S")}',
            'beginTime': date_s.astimezone().replace(microsecond=0).isoformat(),
            'endTime': date_e.astimezone().replace(microsecond=0).isoformat(),
            'cards': cards,
            'faces': [{'faceData': pic_data}]}
    return make_hik_request(url_part, data)


def hik_api_del_person(person_id: str) -> dict:
    url_part = '/artemis/api/resource/v1/person/single/delete'
    data = {'personId': person_id}
    return make_hik_request(url_part, data)


def hik_api_get_person_by_id(person_id: str) -> dict:
    url_part = '/artemis/api/resource/v1/person/personId/personInfo'
    data = {'personId': person_id}
    return make_hik_request(url_part, data)


def hik_api_get_person_by_code(person_code: str) -> dict:
    url_part = '/artemis/api/resource/v1/person/personCode/personInfo'
    data = {'personCode': person_code}
    return make_hik_request(url_part, data)


def hik_api_set_person_face(person_id: str, pic_data: str) -> dict:
    url_part = '/artemis/api/resource/v1/person/face/update'
    data = {'personId': person_id, 'faceData': pic_data}
    return make_hik_request(url_part, data)


def hik_get_all_persons() -> tuple:
    staff_dict = {}
    avrora_dict = {}
    not_staff_dict = {}
    return_result = []
    page_size = 500
    count = 1
    url_part = '/artemis/api/resource/v1/person/personList'
    while True:
        data = {'pageNo': count, 'pageSize': page_size}
        get_data = make_hik_request(url_part, data)
        get_temp = get_data['list']
        current_page = get_data['pageNo']
        if current_page == count:
            return_result.extend(get_temp)
        else:
            break
        count += 1
    logging.debug(f'{count} запросов в HikCentral-api сделано')
    staff_group = (
        GROUP_ID['yandex'], GROUP_ID['external'],
        GROUP_ID[306], GROUP_ID[268],
    )
    for result in return_result:
        if 'email' in result:
            login = result['email'].split('@')[0]
            if result['orgIndexCode'] in staff_group:
                if login in staff_dict:
                    logging.error(f'Dubl_STAFF: {login}: {result}')
                    hik_api_del_person(result['personId'])
                else:
                    staff_dict[login] = result
            elif result['orgIndexCode'] == GROUP_ID[181]:
                if login in avrora_dict:
                    logging.error(f'Dubl_AVR: {login}: {result}')
                    hik_api_del_person(result['personId'])
                else:
                    avrora_dict[login] = result
            else:
                if login in not_staff_dict:
                    logging.error(f'Dubl_OUT: {login}: {result}')
                    hik_api_del_person(result['personId'])
                else:
                    not_staff_dict[login] = result
        else:
            logging.error(f'Нет почты или пропуска: {result}')
    return (staff_dict, avrora_dict, not_staff_dict)


def add_avrora(staff: list) -> list:
    staff_to_avrora = []
    staff_to_avrora.extend(staff)
    staff_to_avrora[3] = staff_to_avrora[4] = 181
    staff_to_avrora[0] = {avrora_badge(tuple(staff[0])[0]): 'avrora badge'}
    return staff_to_avrora


'''def get_vipstaff_badge(vip_level: int = 2) -> dict or None:
    result_dict: dict = {}
    try:
        persons_filter = {
            'person.official.is_robot': False,
            'person.official.affiliation': 'yandex',
            'department_group.level': vip_level,
            'role': 'chief',
            '_fields': ','.join(['person.id', 'person.login', 'person.name',
                                 'person.location.office.id',
                                 'person.official.affiliation']),
        }
        for login, api_data in get_api_staff.staff_api_departments(persons_filter).items():
            result_dict[login] = [
                    {},
                    api_data['person']['name']['first']['ru'],
                    '',  # api_data['person']['name']['last']['ru'],
                    api_data['person']['location']['office']['id'],
                    api_data['person']['official']['affiliation'],
            ]
    except requests.exceptions.SSLError as error:
        logging.error(f'Ошибка соединения со СТАФФ: {error}')
        raise
    return result_dict'''


def hik_upd_picture(login: str, hik_person: dict, updading_photo: dict) -> bool:
    if login in updading_photo:
        person_id = hik_person['personId']
        pic_data = get_photo(PHOTO_DB, login)
        if pic_data:
            logging.info(f'UpdatePIC: {login} (HIK: {person_id}) STAFF: {updading_photo[login]}')
            hik_api_set_person_face(person_id, pic_data)
            return True
    return False


def exclude_wrong_persons(staff_dict: dict) -> dict:
    exclude = 'anirogozina'  # was mistake
    if exclude in staff_dict:
        logging.info(f'Exclude: {exclude}: {staff_dict[exclude]}')
        del staff_dict[exclude]
    return staff_dict


def main() -> None:
    date_s = datetime.datetime.now()
    regions = [
        {'official.staff_biometric_agreement': True},
    ]
    staff_dict = get_staff_logins(regions)
    add_badge = get_allbadge_from_lenel('lenrus')
    logging.info(f'lenel => {len(add_badge)}')
    no_of_badge = set()
    for login in staff_dict:
        if login in add_badge:
            for key in add_badge[login]:
                if key not in staff_dict[login][0] and int(key) > 34000:
                    staff_dict[login][0][key] = 'one badge'
        else:
            no_of_badge.add(login)
    staff_set = set(staff_dict) - no_of_badge
    hik_dict, hik_avrora_dict, hik_outstaff_dict = hik_get_all_persons()
    hik_set = set(hik_dict)

    connection = init_db()
    updading_photo = pictures_for_update(connection, HIKCENTRAL)
    logging.info(f'Всего PICs: {len(updading_photo)}')

    badgewhitelist_set = set()
    get_log_statictic(hik_set, staff_set, badgewhitelist_set)
    for login in hik_set - staff_set - badgewhitelist_set:
        logging.info(f'Delete: {login}: {hik_dict[login]}')
        hik_api_del_person(hik_dict[login]['personId'])
    for login in staff_set - hik_set - badgewhitelist_set:
        if not staff_dict[login][0]:
            logging.error(f'No cards: {login}: {staff_dict[login]}')
            continue
        add_person = hik_api_add_person(login, staff_dict[login])
        if add_person is None:
            if login in hik_outstaff_dict:
                hik_api_upd_person(login, hik_outstaff_dict[login], staff_dict[login])
            else:
                logging.error(f'Duplicale badge!!!: {login}: {staff_dict[login]}')
        else:
            logging.info(f'Add: {login}({add_person}): {staff_dict[login]}')
            do_update(connection, login, HIKCENTRAL)
    for login in (hik_set & staff_set) - badgewhitelist_set:
        hik_api_upd_person(login, hik_dict[login], staff_dict[login])
        if hik_upd_picture(login, hik_dict[login], updading_photo):
            do_update(connection, login, HIKCENTRAL)

    # for Avrora
    logging.info('Делаем АВРОРУ')
    badgewhitelist_set = {
        'le0nko',
        'mayatskiy',
        'irastokin',
        'serge-fedorov',
        'malanichev',
        'dotcenkodd',
        'a-orlyuk',
    }
    hik_set = set(hik_avrora_dict)
    for login in hik_set - staff_set - badgewhitelist_set:
        logging.info(f'Delete Avrora: {login}: {hik_avrora_dict[login]}')
        hik_api_del_person(hik_avrora_dict[login]['personId'])
    for login in staff_set - hik_set - badgewhitelist_set:
        if staff_dict[login][4] == 'yandex' and staff_dict[login][0]:
            add_avrora_person = hik_api_add_person(login, add_avrora(staff_dict[login]))
            if add_avrora_person is None:
                logging.error(f'Duplicate Avrora: {login} {add_avrora(staff_dict[login])}')
            else:
                logging.info(f'Add Avrora: {login}({add_avrora_person}): {add_avrora(staff_dict[login])}')
        else:
            staff_set.remove(login)
    for login in (hik_set & staff_set) - badgewhitelist_set:
        if staff_dict[login][4] == 'yandex':
            hik_api_upd_person(login, hik_avrora_dict[login], add_avrora(staff_dict[login]))
            hik_upd_picture(login, hik_avrora_dict[login], updading_photo)
        else:
            staff_set.remove(login)
    connection.commit()
    connection.close()
    get_log_statictic(hik_set, staff_set, badgewhitelist_set)
    logging.info(f'Затрачено всего: {datetime.datetime.now()-date_s}')
    logging.error(f'Has no any BADGE: {no_of_badge}')
    return None


if __name__ == '__main__':
    prepare_for_logging(LOG_FILE, LOG_FILE_TO_SEND)
    main()
    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'
    send_email(MAIL_TO, 'Выгрузка для HikCentral', body_message, [LOG_FILE_TO_SEND])
    logging.info('----------------------')
