import re
import os
import csv
import requests
import datetime
import hashlib
import get_api_staff
import logging
from file_import_utils import (
    get_secret_from_yav,
    send_email,
    prepare_for_logging,
    get_log_statictic,
    get_allbadge_from_lenel,
)

LOG_FILE = f'{os.path.splitext(__file__)[0]}.log'
LOG_FILE_TO_SEND = 'rps_result.txt'
VAL_UNBLOCKED = 0
VAL_BLOCKED = 1
VAL_MULTIPASS = 1
# LEVEL_DCVLADIMIR = 1
LEVEL_DCSASOVO = 2
LEVEL_00STANDART = 3
LEVEL_NONE = 4

MAIL_TO = ['diff_another@yandex-team.ru']
URL_HOST = 'wiki-api.yandex-team.ru'
YAV_TOKEN = os.getenv('YAV_TOKEN', '')
OAUTH_TOKEN = os.getenv('ROBOT_OAUTH_TOKEN', '')
# OAUTH_TOKEN = get_secret_from_yav('sec-01f35s2537s2d6sed431k7a8kj', 'oauth-token', YAV_TOKEN)
DEVELOPED_ID = get_secret_from_yav('sec-01fn9147jtn6x8j2srnw5ne3hk', 'DEVELOPED_ID', YAV_TOKEN)
KEY_API = get_secret_from_yav('sec-01fn9147jtn6x8j2srnw5ne3hk', 'KEY_API', YAV_TOKEN)
SOURCE_URL = 'parking-vla.r-p-s.ru'
# only for QRcode - not in use at 2021-10-31
COMPANY_ID = 'F23718CF-1E33-1A4E-144E-E542A01BDEFF'
ID_BCM = ''
ID_TP_COUNT = ''
ID_TS_COUNT = ''
TP_RENT = ''
TS_RENT = ''
IMPORT_RFID_FILE = 'RedRoseRFID20211110.csv'
COMPANY_ID_RPS = {
    'yandex': '2239E9F0-8676-3191-56E0-8008E894904D',
    'external': 'D45E6036-789F-0A1B-D807-B39246B54669',
    'deleted': 'B95A0B95-009C-B9C9-A950-76D83721BE4A',
    'wiki-sasovo': 'E9DF3AD7-9464-75DE-B1E6-EE94980BB3D0',
}
DC_FOLDER = {
    'sas': ['users/aslesnyakov/metki-dc-sasovo', 'ДЦ Сасово'],
    'vla': ['users/aslesnyakov/metki-dc-vladimir', 'ДЦ Владимир'],
}
DC_GUEST_FOLDER = {
    'sas-guests':  ['users/ladygin/sasovo/autoguests', 'ДЦ Сасово Гости'],
}


def make_rps_request(url_part: str, request_params: dict = None) -> dict:
    request_params = request_params or {}
    timestamp = int(datetime.datetime.now().timestamp())
    hash_str = hashlib.md5(
        f'{timestamp}{DEVELOPED_ID}{SOURCE_URL}{KEY_API}'.encode('utf-8')
    ).hexdigest()
    payload = {'time': timestamp,
               'developerId': DEVELOPED_ID,
               'hash': hash_str, }
    payload.update(request_params)

    response = requests.post(
        url=f'https://{SOURCE_URL}/api2/remote/{url_part}',
        data=payload,
    )
    if response.status_code != 200:
        response_string = response.content.decode('UTF-8')
        raise RuntimeError(f'{response.status_code}: {response_string}')
    return response.json()


def rps_api_qr_sendmail(qr_id: str) -> dict:
    if not qr_id:
        raise ValueError('Необходимо указывать QRcode ID')
    return make_rps_request('BarCodeInUse/sendEmail', request_params={'Id': qr_id})


def rps_api_qr_inuse(date_start: datetime = None, date_end: datetime = None, offset: int = 0) -> dict:
    date_end = date_end or datetime.datetime.now()
    date_start = date_start or date_end - datetime.timedelta(days=30)
    return make_rps_request(
        'BarCodeInUse',
        request_params={
            'dateStart': date_start,
            'dateEnd': date_end,
            'offset': offset,
        }
    )


def rps_api_qr_add(
        qrid: str = '',
        guest_plate_number: str = '',
        guest_car_model: str = '',
        guest_name: str = 'Incognito',
        guest_email: str = ''
) -> dict or None:
    action = 'update' if qrid else 'create'
    date_end = datetime.datetime.now() + datetime.timedelta(days=3)
    response = make_rps_request(
        f'BarCodeInUse/{action}',
        request_params={
            'Id': qrid,
            'IdBCM': ID_BCM,
            'CompanyId': COMPANY_ID,
            'IdTP': '8',
            'IdTS': '1',
            'DiscountAmount': '',
            'GroupIdFC': '3',
            'UsageTime': date_end.strftime('%Y-%m-%d %H:%M:%S'),
            'GuestPlateNumber': guest_plate_number,
            'GuestCarModel': guest_car_model,
            'GuestName': guest_name,
            'GuestEmail': guest_email,
            'Comment': '',
            'IdTPForCount': ID_TP_COUNT,
            'IdTSForCount': ID_TS_COUNT,
            'TPForRent': TP_RENT,
            'TSForRent': TS_RENT,
            'ExternalId': '',
            'BarCodeDescription': '',
            'CompanyName': 'Яндекс',
        }
    )
    if response['status'] == 'success':
        return response['item']
    return None


def rps_api_listtags(card_id: int) -> dict or None:
    response = make_rps_request('Tags', request_params={'MiFareCardId': card_id})
    if response['status'] == 'success':
        return response['items']
    return None


def rps_api_tag(
        tag_id: str = '',
        card_id: str = '',
        data: str = '----',
        description: str = 'deleted tag',
        tagtype: int = 0
) -> dict or None:
    action = 'update' if tag_id else 'create'
    response = make_rps_request(
        f'Tags/{action}',
        request_params={
            'Id': tag_id,
            'MiFareCardId': card_id,
            'TagId': data,
            'TagType': tagtype,
            'Description': description,
        },
    )
    if response['status'] == 'success':
        return response['item']
    return None


def rps_api_listcards(offset: int = 0) -> dict or None:
    response = make_rps_request('Cards', request_params={'offset': offset})
    if response['status'] == 'success':
        return response['items']
    return None


def rps_api_card(
        card_id: str = '',
        blocked: int = 1,
        client_typid_fc: int = 1,  # 0-onetime, 1-continious
        client_id: str = '',
        company_id: str = '',
        comment: str = '',
        client_group_id: int = 0,
) -> dict or None:
    if card_id:
        action = 'update'
    else:
        action = 'create'
        card_id = str(int(datetime.datetime.now().timestamp()*1000))
    response = make_rps_request(
        f'Cards/{action}',
        request_params={
            'Blocked': blocked,
            'CardId': card_id,
            'ClientId': client_id,
            'CompanyId': company_id,
            'TSidFC': 2,
            'TPidFC': 3,
            'ClientGroupidFC': client_group_id,
            'ClientTypidFC': client_typid_fc,
            'Comment': comment,
        },
    )
    if response['status'] == 'success':
        return response['item']
    return None


def rps_api_listcompanies() -> dict or None:
    response = make_rps_request('Companies')
    if response['status'] == 'success':
        return response['items']
    return None


def rps_api_listclients() -> dict or None:
    response = make_rps_request('Clients')
    if response['status'] == 'success':
        return response['items']
    return None


def rps_api_client(
        client_id: str = '',
        contact_name: str = '',
        phone: str = '',
        email: str = ''
) -> dict or None:
    action = 'update' if client_id else 'create'
    response = make_rps_request(
        f'Clients/{action}',
        request_params={
            'Id': client_id,
            'ContactName': contact_name,
            'Phone': phone,
            'Email': email,
        },
    )
    if response['status'] == 'success':
        return response['item']
    return None


def rps_api_transactions(
        date_start: datetime.datetime = None,
        date_end: datetime.datetime = None,
        limit: int = 0,
) -> dict or None:
    date_end = date_end or datetime.datetime.now()
    date_start = date_start or date_end - datetime.timedelta(days=30)
    response = make_rps_request(
        'Transactions',
        data={
            'dateStart': date_start.strftime('%Y-%m-%d %H:%M:%S'),
            'dateEnd': date_end.strftime('%Y-%m-%d %H:%M:%S'),
            'limit': limit,
        },
    )
    if response.json()['status'] == 'success':
        return response.json()['items']
    return None


def rps_api_listgroups() -> dict or None:
    response = make_rps_request('Groups')
    if response['status'] == 'success':
        return response['items']
    return None


def rps_api_listzones() -> dict or None:
    response = make_rps_request('Zones')
    if response['status'] == 'success':
        return response['items']
    return None


def transform_cars_number(token: str) -> str:
    TRANSFORM: dict = {
            'A': 'А', 'B': 'В', 'C': 'С',
            'E': 'Е', 'H': 'Н', 'K': 'К',
            'M': 'М', 'O': 'О', 'P': 'Р',
            'T': 'Т', 'X': 'Х', 'Y': 'У',
    }
    out_strint: str = ''
    for char in token.strip().replace(" ", "").upper():
        if char in TRANSFORM:
            out_strint += TRANSFORM[char]
        else:
            out_strint += char
    return out_strint


def get_rps_list() -> list:
    return_result = []
    count = 0
    while True:
        get_temp = rps_api_listcards(500*count)
        count += 1
        return_result.extend(get_temp)
        if len(get_temp) < 500:
            break
    logging.debug(f'{count} запросов в RPS_Card-api сделано')
    return return_result


def get_all_rps_list() -> dict:
    rps_clients: dict = {}
    for client in rps_api_listclients():
        rps_clients[client['Id']] = [
                client['Email'].split('@')[0] if client['Email'] else None,
                client['ContactName'],
                client,
        ]
    rps_dict: dict = {}
    for mifare_card in get_rps_list():
        if mifare_card['CompanyId'] in [
            COMPANY_ID_RPS['yandex'],
            COMPANY_ID_RPS['external'],
            COMPANY_ID_RPS['wiki-sasovo']
        ] and mifare_card['Blocked'] == '0':
            if mifare_card['ClientId'] not in rps_clients:
                logging.info(f"DELETE: {mifare_card['ClientId']} {mifare_card}")
                continue
            if rps_clients[mifare_card['ClientId']][0] in rps_dict:
                login = rps_clients[mifare_card['ClientId']][0]
                logging.info(f'!!!!!!!!  MiFare EXIST!!!! dublicate {login}')
                # delete_rps_entity(login, rps_dict[login])
                continue
            tag_dict: dict = {}
            for key in rps_api_listtags(mifare_card['CardId']):
                if key['IsDeleted'] != '1':
                    if key['TagId'] in tag_dict and key['TagId'] != '----':
                        logging.info(f"del duplicate TAG: {rps_clients[mifare_card['ClientId']][0]} {key['TagId']}")
                        rps_api_tag(key['Id'], key['MiFareCardId'], '----', 'deleted tag', 0)
                    else:
                        tag_dict[key['TagId']] = key
            rps_dict[rps_clients[mifare_card['ClientId']][0]] = [
                    mifare_card,
                    rps_clients[mifare_card['ClientId']],
                    tag_dict,
            ]
    return rps_dict


def get_add_badge_from_lenels() -> dict:
    result_dict = get_allbadge_from_lenel('lenrus')
    for key, id_badge in get_allbadge_from_lenel('lenfin').items():
        if key in result_dict:
            result_dict[key].update(id_badge)
        else:
            result_dict[key] = id_badge
    return result_dict


def get_staff_badge(regions: list) -> dict or None:
    result_dict: dict = {}
    for region in regions:
        try:
            persons_filter = {
                '_fields': ','.join(['id', 'login', 'name',
                                     'location.office.id',
                                     'cars.plate',
                                     'cars.model',
                                     'official.affiliation']),
            }
            persons_filter.update(region)
            for login, api_data in get_api_staff.staff_api_persons(persons_filter).items():
                car_dict: dict = {}
                for car in api_data['cars']:
                    car['plate'] = transform_cars_number(car['plate'])
                    if len(car['plate']) > 4:
                        car_dict[car['plate']] = car
                if car_dict:
                    result_dict[login] = [
                            {},
                            api_data['name']['first']['ru'].replace("́", ""),
                            api_data['name']['last']['ru'].replace("́", ""),
                            api_data['location']['office']['id'],
                            car_dict,
                            {}, {}, api_data['official']['affiliation'],
                    ]
        except requests.exceptions.SSLError as error:
            logging.error(f'Ошибка соединения со СТАФФ: {error}')
            return None
    return result_dict


def load_csv_rfid(file_name: str, code_page: str = 'cp1251') -> dict:
    try:
        csv_file = open(file_name, 'r', encoding=code_page)
        csv_reader = csv.reader(csv_file, delimiter=',', quotechar='"')
    except FileNotFoundError:
        logging.info(f'Файл не найден: {file_name}')
        return {}
    data_read = [row for row in csv_reader]
    csv_file.close()
    result_dict: dict = {}
    for _, number_id, status_id, full_name, rfid_id, _, _, _, car_sign, _, _ in data_read:
        if len(rfid_id) > 5:
            normal_car_sign = transform_cars_number(car_sign)
            result_dict[normal_car_sign] = [
                rfid_id,
                normal_car_sign,
                full_name,
            ]
    return result_dict


def get_wiki_rfid(folders: dict) -> dict:
    rfid_data = {}
    rfid_session = requests.Session()
    for folder in folders:
        rfid_session.headers.update({'Authorization': f'OAuth {OAUTH_TOKEN}'})
        rfid_session.headers.update({'Content-Type': 'application/json'})
        response = rfid_session.get(url=f'https://{URL_HOST}/_api/frontend/{folders[folder][0]}/.grid/export/csv')
        if response.status_code != 200:
            raise RuntimeError(f'{response.status_code}: {response.content}')
        csv_reader = csv.reader(response._content.decode().splitlines(), delimiter=',')
        _ = next(csv_reader)
        for row in csv_reader:
            if len(row) > 2:
                normal_car_sign = transform_cars_number(row[1])
                normal_rfid_id = re.sub(r'([^A-F0-9]+)', '', row[2], flags=re.M)
                if bool(normal_car_sign) & bool(row[0]) & bool(normal_rfid_id):
                    rfid_data[normal_car_sign] = [normal_rfid_id, row[1], row[0].split('(')[1].split(')')[0], folder]
    return rfid_data


def get_wiki_sas_guests(folders: dict) -> dict:
    result_dict = {}
    rfid_session = requests.Session()
    for folder in folders:
        rfid_session.headers.update({'Authorization': f'OAuth {OAUTH_TOKEN}'})
        rfid_session.headers.update({'Content-Type': 'application/json'})
        response = rfid_session.get(url=f'https://{URL_HOST}/_api/frontend/{folders[folder][0]}/.grid/export/csv')
        if response.status_code != 200:
            raise RuntimeError(f'{response.status_code}: {response.content}')
        csv_reader = csv.reader(response._content.decode().splitlines(), delimiter=',')
        _ = next(csv_reader)
        for row in csv_reader:
            if len(row) > 4 and row[4] == 'True':
                normal_car_sign = transform_cars_number(row[0])
                result_dict[normal_car_sign] = [
                        {},
                        row[2],
                        row[3],
                        -1,  # from Wiki
                        {normal_car_sign: {'plate': normal_car_sign, 'model': row[1]}},
                        {}, {}, 'wiki-sasovo',
                ]
    return result_dict


def add_rps_entity(name: str, name_list: list) -> None:
    msg_to_log = [f'add {name}']
    if name_list[7] == 'yandex':
        level_id = LEVEL_00STANDART
        comment_id = 'staff_yandex_export'
    elif name_list[7] == 'external':
        level_id = LEVEL_NONE
        comment_id = 'staff_external_export'
    elif name_list[7] == 'wiki-sasovo':
        level_id = LEVEL_DCSASOVO
        comment_id = 'wiki-sasovo_export'
    else:
        logging.warning('No of company_id: %s (%s)', name_list[7], msg_to_log)
        return None
    client_id = rps_api_client(
            '',
            ' '.join([name_list[2], name_list[1], name]),
            'no phone',
            '@'.join([name, 'yandex-team.ru'])
    )['Id']
    company_id = COMPANY_ID_RPS[name_list[7]]
    card_id = rps_api_card(
        '',
        VAL_UNBLOCKED,
        VAL_MULTIPASS,
        client_id,
        company_id,
        comment_id,
        level_id
    )['CardId']
    msg_to_log.append(f'{client_id}, {card_id}')
    for id_key in name_list[0]:
        msg_to_log.append(f'{id_key}')
        rps_api_tag('', card_id, id_key, name_list[0][id_key], 3)
    for id_key in name_list[4]:
        msg_to_log.append(f'{id_key}')
        rps_api_tag('', card_id, id_key, name_list[4][id_key]['model'], 1)
    for id_key in name_list[5]:
        msg_to_log.append(f'{id_key}')
        rps_api_tag('', card_id, id_key, f'RR-RFID: {name_list[5][id_key]}', 2)
    for id_key in name_list[6]:
        msg_to_log.append(f'{id_key}')
        rps_api_tag('', card_id, id_key, f'WIKI-RFID: {name_list[6][id_key]}', 2)
    logging.info(', '.join(msg_to_log))
    return None


def delete_rps_entity(name: str, name_list: list) -> None:
    # Block and moving to 'deleted'
    date_block = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    if not name:
        name = 'None'
    logging.info(f"delete {name}")
    rps_api_card(
        name_list[0]['CardId'],
        VAL_BLOCKED,
        VAL_MULTIPASS,
        name_list[0]['ClientId'],
        COMPANY_ID_RPS['deleted'],
        ' '.join(['Deleted:', date_block]),
        LEVEL_NONE
    )
    if name_list[0]['ClientId'] != 'None':
        rps_api_client(name_list[0]['ClientId'], ' '.join(['deleted', name]), 'None', 'None')
    for tag in name_list[2]:
        logging.info(f' -> delete TAG: {tag}')
        rps_api_tag(name_list[2][tag]['Id'], name_list[2][tag]['MiFareCardId'], '----', 'deleted tag', 0)
    return None


def update_rps_entity(login: str, rps: dict, staff: dict) -> None:
    card_id = rps[0]['CardId']
    staff_set = set(staff[0])
    staff_set.update(staff[4])
    staff_set.update(staff[5])
    staff_set.update(staff[6])
    rps_tag_set = set(rps[2])
    deleted_entity_for_add = set()
    for del_entity in rps_tag_set-staff_set:
        if rps[2][del_entity]['TagId'] != '----':
            logging.info(f'del tag from rps {login} {del_entity}')
            rps_api_tag(rps[2][del_entity]['Id'], card_id, '----', 'deleted tag', 0)
        deleted_entity_for_add.add(del_entity)
    for add_entity in staff_set-rps_tag_set:
        if deleted_entity_for_add:
            tag_id = rps[2][deleted_entity_for_add.pop()]['Id']
        else:
            tag_id = ''
        if add_entity in staff[0]:
            logging.info(f'add BADGE from staff {login} {add_entity} {staff[0][add_entity]}')
            rps_api_tag(tag_id, card_id, add_entity, staff[0][add_entity], 3)
        elif add_entity in staff[4]:
            logging.info(f'add GNZ from staff {login} {add_entity}')
            rps_api_tag(tag_id, card_id, add_entity, staff[4][add_entity]['model'], 1)
        elif add_entity in staff[5]:
            logging.info(f'add RR-RFID from staff(redrose) {login} {add_entity}')
            rps_api_tag(tag_id, card_id, add_entity, f'RR-RFID: {staff[5][add_entity]}', 2)
        elif add_entity in staff[6]:
            logging.info(f'add WIKI-RFID from WIKI {login} {add_entity}')
            rps_api_tag(tag_id, card_id, add_entity, f'WIKI-RFID: {staff[6][add_entity]}', 2)
        else:
            logging.error(f'!! no of TAG: {login} {add_entity}')
    return None


def main() -> None:
    date_s = datetime.datetime.now()
    regions = [
        {'official.affiliation': 'yandex'},
        {'login': 'lbeygul'},
    ]
    staff_dict = get_staff_badge(regions)
    logging.info(f'staff => {len(staff_dict)}')
    if not staff_dict:
        logging.error('No staff!')
        return None
    add_badge = get_add_badge_from_lenels()
    logging.info(f'lenel => {len(add_badge)}')
    rfid_tikets = load_csv_rfid(IMPORT_RFID_FILE)
    logging.info(f'rfid => {len(rfid_tikets)}')
    rfid_wiki = get_wiki_rfid(DC_FOLDER)
    logging.info(f'wiki => {len(rfid_wiki)}')
    for login in staff_dict:
        if login in add_badge:
            for key in add_badge[login]:
                if key not in staff_dict[login][0]:
                    staff_dict[login][0][key] = 'add badge'
        for key_sign in staff_dict[login][4]:
            if key_sign in rfid_tikets:
                staff_dict[login][5][rfid_tikets[key_sign][0]] = key_sign
            if key_sign in rfid_wiki:
                staff_dict[login][6][rfid_wiki[key_sign][0]] = key_sign

    staff_dict.update(get_wiki_sas_guests(DC_GUEST_FOLDER))
    staff_set = set(staff_dict)
    rps_dict = get_all_rps_list()
    rps_set = set(rps_dict)
    badgewhitelist_set = set()
    get_log_statictic(rps_set, staff_set, badgewhitelist_set)

    for login in (staff_set-rps_set)-badgewhitelist_set:
        add_rps_entity(login, staff_dict[login])
    for login in (rps_set-staff_set)-badgewhitelist_set:
        delete_rps_entity(login, rps_dict[login])
    for login in (rps_set & staff_set)-badgewhitelist_set:
        update_rps_entity(login, rps_dict[login], staff_dict[login])
    logging.info(f'Затрачено всего: {datetime.datetime.now()-date_s}')
    return None


if __name__ == '__main__':
    prepare_for_logging(LOG_FILE, LOG_FILE_TO_SEND)
    main()
    body_message = 'Файлы приложены как вложения к данному письму. '\
        'Письмо было сгенерировано роботом, пожалуйста, не отвечайте на него.<br><br>'
    send_email(MAIL_TO, 'Выгрузка для R-P-S', body_message, [LOG_FILE_TO_SEND])
    logging.info('----------------------')
