import logging
from collections import defaultdict

from django.http import Http404
from ninja import NinjaAPI
from ninja.errors import ValidationError as NinjaValidationError

from intranet.paste.src.api_v1.exceptions import NinjaApiError, InternalServerError, ValidationError, NotFound

logger = logging.getLogger(__name__)


def subscribe_default_exception_handlers(api: NinjaAPI):
    @api.exception_handler(NinjaApiError)
    def ninja_api_error(request, exc: NinjaApiError):
        return api.create_response(
            request,
            exc.to_dict(),
            status=exc.status_code,
        )

    @api.exception_handler(Exception)
    def any_error(request, exc):
        logger.exception(f'Unhandled exception in API: {exc} ({exc.__class__})')
        return ninja_api_error(request, InternalServerError(debug_message=f'{exc} ({exc.__class__})'))

    @api.exception_handler(NinjaValidationError)
    def validation_error(request, exc: NinjaValidationError):
        details = defaultdict(lambda: defaultdict(list))

        for error in exc.errors:
            source = error['loc'][0]
            if len(error['loc']) > 2:
                field_name = error['loc'][2]
            else:
                field_name = error['loc'][1]

            error_msg = {'error_code': error['type'], 'debug_message': error['msg'], 'details': {}}

            if len(error['loc']) > 3:
                error_msg['details']['index'] = error['loc'][3]

            if context := error.get('ctx', None):
                error_msg['details']['context'] = context

            details[source][field_name].append(error_msg)

        return ninja_api_error(request, ValidationError(details=details))

    @api.exception_handler(Http404)
    def not_found_error(request, exc):
        return ninja_api_error(request, NotFound())
