from typing import Any, Dict


class NinjaApiError(Exception):
    status_code: int = 500
    error_code: str = 'API_ERROR'
    debug_message: str = 'Internal api error'
    details: Dict[str, Any] = None

    def __init__(self, debug_message=None, details=None, error_code=None):
        if error_code is not None:
            self.error_code = error_code

        if debug_message is not None:
            self.debug_message = debug_message

        if details is not None:
            self.details = details

    def to_dict(self) -> Dict[str, Any]:
        return {'error_code': self.error_code, 'debug_message': self.debug_message, 'details': self.details}

    @classmethod
    def from_legacy_error(cls, legacy_error: 'RestApiError'):
        error_details = []
        if legacy_error.errors:
            error_details.append(
                '; '.join(field + ': ' + ', '.join(errors) for field, errors in legacy_error.errors.items())
            )
        if legacy_error.non_field_messages:
            error_details.append(','.join(legacy_error.non_field_messages))

        if legacy_error.details:
            error_details.append(legacy_error.details)

        error_details = '| '.join(error_details)

        err = cls(error_code=legacy_error.error_code, details=error_details, debug_message=legacy_error.debug_message)
        err.status_code = legacy_error.status_code

        if err.status_code == 409:  # в старой апи 409 - ошибка валидации. тут это типичный 400
            err.status_code = 400

        return err


class ValidationError(NinjaApiError):
    error_code = 'VALIDATION_ERROR'
    debug_message = 'Validation failed'
    status_code = 400


class InternalServerError(NinjaApiError):
    error_code = 'INTERNAL_SERVER_ERROR'
    debug_message = ''
    status_code = 500


class NotAllowed(NinjaApiError):
    error_code = 'METHOD_NOT_ALLOWED'
    debug_message = 'Method not allowed'
    status_code = 405


class BadRequest(NinjaApiError):
    error_code = 'BAD_REQUEST'
    debug_message = ''
    status_code = 400


class FeatureNotImplemented(NinjaApiError):
    error_code = 'NOT_IMPLEMENTED'
    debug_message = ''
    status_code = 400


class QuotaExceeded(NinjaApiError):
    error_code = 'QUOTA_EXCEEDED'
    debug_message = ''
    status_code = 400


class Forbidden(NinjaApiError):
    error_code = 'FORBIDDEN'
    debug_message = ''
    status_code = 403


class NotFound(NinjaApiError):
    error_code = 'NOT_FOUND'
    debug_message = ''
    status_code = 404


class ParamsNeeded(NinjaApiError):
    error_code = 'PARAMS_NEEDED'
    debug_message = 'You should specify some params'


class AlreadyExists(NinjaApiError):
    error_code = 'ALREADY_EXISTS'
    debug_message = 'This resource already exists'
    status_code = 400
