from typing import List, TypeVar, Generic, Callable, Iterable

from ninja import Schema
from pydantic.generics import GenericModel
from pydantic import conint, Field
from django.core.paginator import Paginator

T = TypeVar('T')
T2 = TypeVar('T2')


class PaginationQuery(Schema):
    page_id: conint(ge=1) = Field(default=1)
    page_size: conint(ge=1, le=50) = Field(default=25)


class Collection(GenericModel, Generic[T]):
    results: List[T]
    has_next: bool

    @classmethod
    def build(
        cls,
        qs,
        serializer: Callable[[T2], T],
        filter: Callable[[List[T2]], Iterable[T2]] = None,
        pagination: PaginationQuery = None,
    ) -> 'Collection[T]':
        page_size = pagination.page_size
        page_id = pagination.page_id

        p = Paginator(qs, page_size)
        paginated_results = p.get_page(page_id)

        objects = paginated_results.object_list
        if filter:
            objects = filter(objects)

        return Collection(
            results=[serializer(orm) for orm in objects],
            has_next=paginated_results.has_next(),
        )


class Empty(Schema):
    pass


class StatusOk(Schema):
    status: str = 'ok'


DELETED = 204, ''
DELETED_RESPONSE = {204: Empty}
