import logging
import uuid

import nbformat
from django.contrib.auth.models import User
from django.db import models
from django.db.utils import IntegrityError
from django.urls import reverse
from django.utils.html import escape
from nbconvert import HTMLExporter
from pygments import highlight
from pygments.formatters import HtmlFormatter
from pygments.lexers import ClassNotFound

from intranet.paste.src.coreapp.consts import SYNTAX_LABELS
from intranet.paste.src.coreapp.styles import YaDarkStyle, YaLightStyle
from intranet.paste.src.utils.fields import CompressedTextField
from intranet.paste.src.utils.lexers import get_lexer_by_name

FRAGMENT_LINES_COUNT = 5

RENDER_AS_PLAIN_FOR_FRAGMENT = {
    'ipynb',
}

REPLACE_LIB_MAP = {
    'https://cdnjs.cloudflare.com/ajax/libs/require.js/2.1.10/require.min.js': 'https://yastatic.net/s3/frontend/jslibs/require.js/2.1.10/require.min.js',
    'https://cdnjs.cloudflare.com/ajax/libs/jquery/2.0.3/jquery.min.js': 'https://yastatic.net/jquery/2.0.3/jquery.min.js',
    'https://cdnjs.cloudflare.com/ajax/libs/mathjax/2.7.5/latest.js': 'https://yastatic.net/mathjax/2.7.5/mathjax.js',
    # 'https://yastatic.net/s3/frontend/jslibs/mathjax/2.7.5/mathjax.js'
}


def to_html(text, syntax='plain', noclasses=False, style=None):
    try:
        if style:
            if style == 'ya_light':
                style = YaLightStyle
            elif style == 'ya_dark':
                style = YaDarkStyle
            else:
                style = 'default'
        else:
            style = 'default'
        if syntax == 'ipynb':
            html_exporter = HTMLExporter()

            body, _resources = html_exporter.from_notebook_node(nbformat.reads(text, as_version=4))
            for outer_lib, inner_lib in REPLACE_LIB_MAP.items():
                body = body.replace(outer_lib, inner_lib)
            return body
        formatter = HtmlFormatter(style=style, noclasses=noclasses)
        return highlight(
            text,
            get_lexer_by_name(syntax),
            formatter,
        )
    except ClassNotFound:
        return '<div class="highlight"><pre>%s\n</pre></div>' % escape(text)


class Code(models.Model):
    objects: models.Manager
    id: int

    SYNTAX_CHOICES = [(k.value, v) for k, v in SYNTAX_LABELS.items()]

    author = models.ForeignKey(User, null=True, blank=True, editable=False, on_delete=models.SET_NULL)
    syntax = models.CharField(max_length=64, choices=SYNTAX_CHOICES, db_index=True)
    text = CompressedTextField()

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    is_deleted = models.BooleanField(default=False)
    is_compressed = models.BooleanField(default=False)

    uuid = models.UUIDField(db_index=True, blank=True, null=True, editable=False)

    def assign_uuid_and_save(self, attempts=3):
        while attempts > 0:
            self.uuid = uuid.uuid4()
            try:
                self.save()
                return
            except IntegrityError:
                logging.exception('uuid4 collision, seriously?')

            attempts -= 1

        raise ValueError('uuid4 collision')

    def get_absolute_url(self):
        return reverse('paste:code', kwargs={'pk': self.pk})

    @property
    def html(self):
        return to_html(self.text, self.syntax)

    @property
    def inline_html(self):
        return to_html(self.text, self.syntax, noclasses=True)

    def to_html(self, noclasses=True, style=None):
        return to_html(self.text, self.syntax, noclasses=noclasses, style=style)

    def fragment(self):
        if isinstance(self.text, bytes):
            splitter = b'\n'
        else:
            splitter = '\n'

        lines = self.text.split(splitter)[:FRAGMENT_LINES_COUNT]
        syntax = self.syntax
        if syntax in RENDER_AS_PLAIN_FOR_FRAGMENT:
            syntax = 'plain'
        return to_html(splitter.join(lines), syntax)

    def __repr__(self):
        return f'<Code #{self.uuid} (pk={self.pk}) [{self.syntax}]>'

    def __str__(self):
        return self.__repr__()

    class Meta:
        db_table = 'paste_code'
