import logging

from django.http import HttpResponse
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.models import User
from django.contrib.sites.shortcuts import get_current_site

from intranet.paste.src.coreapp.models import Code, to_html
from intranet.paste.src.coreapp.forms import CodeForm
from intranet.paste.src.utils.collablogger.experiments import is_experiment, Experiment, get_post_url

logger = logging.getLogger(__name__)


def can_user_delete(user, code):
    return user == code.author or user.has_perm('ypaste.delete_code')


def get_code(pk=None, uuid=None, extra_filter=None):
    extra_filter = extra_filter or {}
    if pk:
        return get_object_or_404(Code, pk=pk, is_deleted=False, **extra_filter)
    elif uuid:
        return get_object_or_404(Code, uuid=uuid, is_deleted=False, **extra_filter)
    else:
        raise ValueError()


def index(request):
    if request.method == 'POST' and request.is_ajax():
        syntax = request.POST.get('syntax', 'plain')
        text = request.POST.get('text')
        style = request.POST.get('style')
        return HttpResponse(to_html(text, syntax, style=style))

    form = CodeForm(request.POST or None)

    if form.is_valid():
        if request.is_service_readonly:
            return redirect('index')

        code = form.save(commit=False)
        if request.user.is_authenticated:
            code.author = request.user
        code.is_compressed = True
        code.assign_uuid_and_save()

        logger.info(f'{request.user} created paste {code}')

        return redirect('code', pk=code.pk)

    return render(request, 'coreapp/index.html', {'form': form})


def code(request, pk=None, uuid=None):
    code = get_code(pk=pk, uuid=uuid)

    style = request.GET.get('style')
    context = {
        'code': code,
        'domain': get_current_site(request).domain,
        'html': to_html(
            text=code.text,
            syntax=code.syntax,
            noclasses=True,
            style=style,
        ),
        'style': style,
        'can_user_delete': can_user_delete(request.user, code),
        'use_uuid': uuid is not None,
    }
    return render(request, 'coreapp/code.html', context)


def html(request, pk=None, uuid=None):
    code = get_code(pk=pk, uuid=uuid)

    return HttpResponse(
        code.inline_html,
        content_type='text/html; charset=utf-8',
    )


def text(request, pk=None, uuid=None):
    code = get_code(pk=pk, uuid=uuid)
    return HttpResponse(
        code.text,
        content_type='text/plain; charset=utf-8',
    )


def delete(request, pk=None, uuid=None):
    code = get_code(pk=pk, uuid=uuid)

    if can_user_delete(request.user, code):
        code.is_deleted = True
        code.save()

    return HttpResponse('OK', content_type='text/plain; charset=utf-8')


def last_code(request):
    if is_experiment(Experiment.FORBID_LAST):
        return render(request, 'coreapp/noway.html', {'author': None, 'post_url': get_post_url()}, status=403)

    context = {
        'author': None,
        'codes': (Code.objects.filter(is_deleted=False).select_related().order_by('-pk')[:25]),
    }
    return render(request, 'coreapp/last.html', context)


def user_code(request, username):
    author = get_object_or_404(User, username=username)

    if author != request.user and is_experiment(Experiment.FORBID_HOMEPAGE):
        return render(request, 'coreapp/noway.html', {'author': None, 'post_url': get_post_url()}, status=403)

    context = {
        'author': author,
        'codes': (Code.objects.filter(author=author, is_deleted=False).order_by('-pk')[:25]),
    }
    return render(request, 'coreapp/last.html', context)


def ok_200(request):
    return HttpResponse('OK', content_type='text/plain; charset=utf-8')
