import re
from django.utils.deprecation import MiddlewareMixin
import waffle
from django.conf import settings
from django.db import DEFAULT_DB_ALIAS
from django.http import HttpResponse
from django_replicated.dbchecker import db_is_alive
from django_replicated.utils import routers


class ReadOnlyMiddleware(MiddlewareMixin):
    def process_request(self, request):
        if request.path.startswith('/admin') or request.path.startswith('/ping'):
            request.is_service_readonly = False
            return

        if waffle.flag_is_active(request, 'is_readonly'):
            request.is_service_readonly = True
        else:
            request.is_service_readonly = not db_is_alive(
                db_name=DEFAULT_DB_ALIAS,
                cache_seconds=getattr(settings, 'READ_ONLY_DOWNTIME', 20),
                number_of_tries=getattr(settings, 'READ_ONLY_TRIES', 1),
            )

        if request.is_service_readonly:
            routers.use_state('slave')

    def process_response(self, request, response):
        """
        Если роутер был перенаправлен на слейв из-за флага read-only,
        возвращаем все на место
        """
        if getattr(request, 'is_service_readonly', False):
            routers.revert()

        return response


class BanMiddleware(MiddlewareMixin):
    def process_request(self, request):
        if request.META.get('HTTP_USER_AGENT'):
            agent = request.META['HTTP_USER_AGENT'].lower()
            if agent.startswith('go-http-client/'):
                return self.ban('https://st.yandex-team.ru/COMPUTERPROBLEM-223')

    def ban(self, reason):
        return HttpResponse('you are banned, {}'.format(reason))


class PingMiddleware(MiddlewareMixin):
    ping_re = re.compile(r'^/ping/?$')

    def process_request(self, request):
        if self.ping_re.match(request.path):
            return HttpResponse('OK')
