from typing import List

from pydantic import (
    BaseSettings,
    Field,
)


class DbHosts(str):
    # строка вида "dc1:host1,dc2:host2"
    @classmethod
    def __get_validators__(cls):
        yield cls.validate

    @classmethod
    def validate(cls, v):
        if not isinstance(v, str):
            raise TypeError('string required')

        db_hosts = []
        for db_host_str in v.split(','):
            parts = db_host_str.split(':')
            db_hosts.append(
                (
                    ''.join(parts[1:]),
                    parts[0],
                )
            )
        return db_hosts


class DbSettings(BaseSettings):
    name: str = Field(env='DB_NAME')
    user: str = Field(env='DB_USER')
    password: str = Field(env='DB_PASSWORD')
    port: int = Field(env='DB_PORT')
    hosts: DbHosts = Field(env='DB_HOSTS')


class Tvm2Settings(BaseSettings):
    client_id: str = Field(env='PASTE_TVM2_CLIENT_ID')
    secret: str = Field(env='PASTE_TVM2_SECRET')
    allowed_clients: List[str] = Field(env='ALLOWED_CLIENTS', default_factory=list)


class Settings(BaseSettings):
    secret: str

    debug: bool = Field(default=False, env='DJANGO_DEBUG')

    hostname: str = Field(default='paste.yandex-team.ru', env='PASTE_HOST')
    loglevel: str = Field(default='INFO', env='DJANGO_LOG_LEVEL')
    allowed_hosts: List[str] = Field(default_factory=list)

    db: DbSettings = DbSettings()
    tvm2: Tvm2Settings = Tvm2Settings()

    unittest: bool = Field(default=False, env='TESTS')
    use_migrations_in_tests: bool = Field(env='USE_MIGRATIONS', default=False)


TYPED_SETTINGS = Settings()
