import logging
import time

from ylog.context import log_context

from .consts import CollabLoggerConfig
from .contexts import extract_normalized_route, build_auth, build_request

from django.conf import settings

logger = logging.getLogger('collablogger')

if hasattr(settings, 'COLLAB_LOGGER_CONFIG'):
    CFG = settings.COLLAB_LOGGER_CONFIG
else:
    CFG = CollabLoggerConfig()

SPAMMY_ROUTE_SET = frozenset(CFG.spammy_routes or [])


class CollabLoggerMiddleware:
    """
    Yet another django logger
    """

    def __init__(self, get_response=None):
        self.get_response = get_response

    def __call__(self, request):
        start_time = time.time()
        route = extract_normalized_route(request)

        auth_ctx = None
        request_ctx = None

        try:
            auth_ctx = build_auth(request, CFG)
            request_ctx = build_request(request, CFG)
        except Exception:  # noqa
            logger.exception('Cant gather request details')

        with log_context(
            route=route,
            auth=auth_ctx,
            request=request_ctx,
        ):
            try:
                response = self.get_response(request)
            except Exception:  # noqa
                logger.exception('[500] Unhandled Exception %s %s' % (request.method, request.get_full_path()))
                raise

            execution_time_ms = int((time.time() - start_time) * 1000)

            with log_context(
                execution_time=int(execution_time_ms),
                status_code=response.status_code,
            ):
                self._log_response(request, response, route, execution_time_ms)

        return response

    def _log_response(self, request, response, route, execution_time_ms):
        log_level = logging.DEBUG

        prompt = '[%s] %s %s' % (response.status_code, request.method, request.get_full_path())
        timeshaming = ''

        if CFG.do_timeshaming:
            if execution_time_ms > CFG.slow_err_ms:
                log_level = logging.ERROR
                timeshaming = ' - Close to timeout'
            elif execution_time_ms > CFG.slow_warn_ms:
                log_level = logging.WARN
                timeshaming = ' - Slow'

        if response.status_code >= 500:
            log_level = logging.ERROR

        ignored = log_level == logging.DEBUG and route in SPAMMY_ROUTE_SET

        if not ignored:
            logger.log(log_level, f'{prompt}{timeshaming}')
